;-----------------------------------------------------------------------
; Usage:        [LH] DOSMON program [arguments]
;
; DOSMON is a quick and dirty DOS monitor for those of us who lost their
; "Undocumented DOS" INTRSPY.  DOSMON shows DOS interrupts on stderr of
; the monitored program; this permits to redirect stderr output.  DOSMON
; uses INT 29h ("fast putchar" screen output) if DOS is busy.
;
; INT 21h stdin, stdout, or stderr i/o functions are silently ignored;
; this includes functions 3Fh, 40h, 4406h, and 4407h for handles 0..2.
; Any stdaux or stdprn (handles 3..4) i/o is shown.
;
; The exit functions 00h, 31h, and 4Ch do not return, only entry values
; are shown.  Exit values apparently corrresponding to function 4Ch or
; 0 actually belong to the exec function 4Bh that started the terminated
; program.
;
; DOSMON expects that a RETF 0CBh at CS:[52h] on entry is still a RETF
; on return from INT 21h functions.  This simplifies the monitoring of
; functions 26h (copy PSP) and 55h (create PSP), where the CS in CS:IP
; return addresses should be the PSP of the caller.  If DOSMON finds no
; RETF at CS:[52h] it calls the old INT 21h directly.
;
; The normal COM-version (DOSMON.COM) needs 1696 bytes for itself when
; it spawns the child process.                  (2010, Frank Ellermann)
;-----------------------------------------------------------------------

_TEXT    segment use16 para public 'CODE'       ;para permits align 16
         assume cs:_TEXT, ds:DGROUP, es:PSP, ss:nothing
         extrn  _edata  : byte  ;end of _DATA, start of _BSS
         extrn  _end    : byte  ;end of _BSS, start of STACK
        .386                    ;allow .386 instructions (lss, etc.)

        ifndef  __TINY__        ;-------------- EXE setup --------------
DGROUP   group  CONST, STRINGS, _DATA, DATA, _BSS, STACK
CSGROUP  equ    DGROUP          ;CS for __TINY__, else DGROUP
PSP      equ    nothing         ;placeholder for assume

main     proc   far             ;program entry point
         xchg   ax,si           ;save initial AX in SI
         mov    bx,CSGROUP      ;setup data segment DS
         mov    ds,bx
         mov    ax,ss
         sub    ax,bx           ;align SS with DGROUP
         jc     _start0         ;SS < DGROUP rejected
         cmp    ah,15           ;keep far stack as is:
         ja     _start1         ;SS > DGROUP + 0FFFh
         shl    ax,4            ;DGROUP offset bottom
         add    ax,sp           ;DGROUP offset top
         jc     _start1         ;keep far SS:SP as is
         mov    ss,bx           ;set SS before SP
        else                    ;-------------- COM setup --------------
         org    100h
DGROUP   group  _TEXT, CONST, STRINGS, _DATA, DATA, _BSS
CSGROUP  equ    cs              ;CS for __TINY__, else DGROUP
PSP      equ    DGROUP          ;placeholder for assume
STACK    equ    512             ;wanted stack size (after _end)

main     proc   near            ;program entry point 100h for __TINY__
         xchg   ax,si           ;save initial AX in SI
         mov    ax,offset _end
         add    ax,STACK
         jc     _start0         ;fatal error if 64 KB DGROUP exhausted
         cmp    sp,ax
         jb     _start0         ;SP too small (loaded at end of memory)
         and    al,-2           ;align 2: _end can be odd
        endif                   ;---------------------------------------
         mov    sp,ax           ;adjust near stack (after SS)
_start1: mov    ax,ss           ;AX:BX = SS:SP (end of stack)
         mov    bx,sp
         mov    dx,es           ;DX = ES (begin at PSP)
         sub    ax,dx           ;paragraphs ES:0 to SS:0
         dec    bx
         shr    bx,4            ;paragraphs SS:0 to SS:SP
         inc    bx
         add    bx,ax           ;paragraphs ES:0 to SS:SP

         mov    ax,4A00h        ;4Ah: reallocate memory
         mov    cx,offset _end
         mov    di,offset _edata
         sub    cx,di           ;_BSS size (can be empty)
        ifndef  __TINY__        ;-------------- EXE setup --------------
         push   ds
         pop    es              ;ES:DI = _edata
         rep    stosb           ;clear _BSS below _end
         mov    es,dx           ;reset ES = PSP segment
        else                    ;-------------- COM setup --------------
         rep    stosb           ;ES:DI = _edata, clear _BSS below _end
        endif                   ;---------------------------------------
         push   cx              ;mark stack frame with 0
         mov    bp,sp           ;initialize BP for debug
         int    21h             ;trim (size BX, segment ES)
         jnc    _start_         ;NC: okay, C: memory error
_start0: mov    ax,4C09h        ;rc 9 invalid memory block
         int    21h
_start_: xchg   ax,si           ;end of startup prologue

; AX = yyxx (FCB2 yy, FCB1 xx: 00 okay, FFh invalid)
; CX = 0000  ,  BP = SP ,  SS:[BP] = 0000 (COM RETN)
; DI = _end  ,  DX = PSP,  BX = DGROUP paragraphs
; DS = DGROUP,  ES = PSP,  SS = DGROUP or far stack
; FS = n/a   ,  GS = n/a,  SI = PSP (undocumented ?)
;-----------------------------------------------------------------------
;        handle arguments and prepare exec parameters

         xchg   ax,dx           ;AX = PSP
         mov    word ptr fcb1+2,ax
         mov    word ptr fcb2+2,ax
         mov    ax,5ch          ;FCB1 at PSP:5Ch
         mov    word ptr fcb1+0,ax
         add    al,10h          ;FCB2 at PSP:6Ch
         mov    word ptr fcb2+0,ax
        ifndef  __TINY__        ;-------------- EXE PSP ----------------
         push   es              ;save PSP for PSP:DTA copy
        endif                   ;---------------------------------------

         xchg   cx,es:[002Ch]   ;PSP offset 2Ch environment
         mov    es,cx           ;own environmemt segment CX
         mov    execpar,cx      ;note envseg CX for EXEC
         assume es:nothing      ;assume no reallocation before EXEC:
         mov    ah,49h          ;free memory block in ES
         int    21h             ;child can get the block
         jc     _start0         ;use startup DOS error exit

         call   initpath        ;find PATH in environment (sets AX 0)
         mov    dh,'?'          ;DX for option /? or -?, DL switchar

DTA      equ    80h             ;PSP:80h is default DTA
         push   ds
         pop    es              ;target ES data segment
        ifndef  __TINY__        ;-------------- EXE PSP ----------------
         pop    ds              ;source DS in saved PSP
        endif                   ;---------------------------------------
         assume ds:PSP, es:DGROUP
         mov    si,DTA          ;source DS:SI = PSP:80h
         mov    di,offset comline
         push   di              ;DI used twice below...
         push   di              ;target ES:DI (comline)
         mov    cx,si           ;copy arguments line for
         rep    movsb           ;use as external command
         pop    di              ;...comline buffer at DI
         pop    si              ;DI = SI

        ifndef  __TINY__        ;-------------- EXE data ---------------
         push   es
         pop    ds
        endif                   ;---------------------------------------
         assume ds:DGROUP
         lodsb                  ;DI < SI, AH = 0 by initpath
         xchg   ax,cx           ;CX = length argument string

testarg: jcxz   noarg           ;any character left ?
         dec    cx              ;remaining length - 1
         lodsb
         cmp    al,' '          ;skip controls & ' '
         jbe    testarg         ;blank, tab, lf, etc.

         inc    cx              ;remaining length
         dec    si              ;1st char. above ' '
         cmp    [si],dx         ;DX for option /? or -?
         jne    cominc

noarg:   mov    dx,offset usage1
         call   message
         mov    dx,offset usage2
         mov    al,1            ;rc 1 nothing happened

exiterr: call   message
         mov    dx,offset crlf
         jmp    exitmsg         ;terminating with return code AL

cominc:  cmp    byte ptr [si],20h
         jbe    comgot          ;copy 1st argument to
         movsb                  ;command name buffer
         loop   cominc

comgot:  mov    [di],ch         ;insert NUL terminator
         lea    bx,[si-1]       ;rest length of arguments
         mov    [bx],cl         ;to new command line size
         mov    word ptr execpar+02,bx

         les    di,fcb1         ;PSP offset 5Ch is 1st FCB
         assume es:PSP
         mov    ax,2901h        ;parse 2nd argument to FCB
         int    21h             ;FCB option 1: skip leading separators
         add    di,10h          ;PSP offset 6Ch is 2nd FCB
         mov    al,1            ;parse 3rd argument to FCB
         int    21h

;-----------------------------------------------------------------------
;        parse command to file name

        ifndef  __TINY__        ;-------------- EXE data ---------------
         push   ds
         pop    es
        endif                   ;---------------------------------------
         assume es:DGROUP
         mov    dx,offset comline
         call   strlen          ;length of command name
         mov    di,dx           ;scan end of path in DI
         mov    al,cwdpath+1

haspath: mov    si,di           ;keep end of path in SI
         jcxz   endpath         ;skip specified path
         repne  scasb           ;scan path character AL
         jz     haspath

endpath: xchg   dx,si           ;keep end of path in DX
         mov    cx,dx           ;determine length in CX
         sub    cx,si           ;path length excl. command

         mov    di,offset com   ;parse '????????.COM',0
         call   comorexe        ;or explicit extension
         lahf
         mov    di,offset exe   ;parse '????????.EXE',0
         call   comorexe        ;or explicit extension
         je     search          ;EXE, CX preserved
         sahf
         je     search          ;COM (or no extension)

nowhere: call   near whoami     ;command not found
         mov    di,offset com
         mov    si,offset exe
         xor    bx,bx
         cmp    byte ptr [si+1],':'
         je     indrive         ;show specified drive
         mov    bl,2            ;skip dummy CWD path
indrive: lea    dx,[bx+di]
         add    bx,si
         mov    cx,7            ;7 * 2 = 2 + 8 + 1 + 3
         repe   cmpsw
         jcxz   badext          ;if explicit extension
         push   bx
         call   message         ;COM
         mov    dx,offset misscom
         call   message         ;or
         pop    dx
badext:  mov    al,2            ;EXE
         jmp    exiterr

whoami:  mov    dx,offset dostool
         call   message
         mov    dx,offset missing
         call   message         ;command not found
         retn                   ;near inline proc

;-----------------------------------------------------------------------
;        search command (given type or COM / EXE, given path or PATH)

search:  mov    bx,offset command
         mov    di,bx           ;command file target
         jcxz   nopath

         mov    si,offset comline
         rep    movsb           ;copy specified path
         call   doithere        ;try COM or EXE path
         jnc    doit            ;use path at DS:DX

         push   bx
         call   near whoami     ;command not found
         pop    dx              ;with specified path
         jmp    exiterr         ;keep error AL as is

nopath:  call   doit_cwd        ;try COM or EXE in
         jnc    doit            ;current directory
         cmp    byte ptr [bx+1],':'
         je     nowhere         ;drive is specified

nothere: call   nextpath        ;DI: next PATH token
         jc     nowhere         ;command file target
         call   doithere        ;try COM or EXE path
         jc     nothere         ;try next PATH token

doit:    jmp    exec            ;minimize used memory
main     endp                   ;(overwrite own code)

;-----------------------------------------------------------------------
comorexe proc   near            ;parse name DX to filename.ext DI
         push   ax              ;keeps AX and CX, expects CH = 0
         push   cx              ;modifies DI, SI
         push   di
         inc    di              ;FCB parse com+1 or exe+1 unused
         mov    si,dx
         mov    ax,2908h        ;parse command DS:SI to ES:DI,
         int    21h             ;FCB option 8: keep extension
         or     al,al
         jz     parseok         ;no '*' or '?', no bad drive
         pop    di              ;reset DI
         jmp    short parserr   ;NZ: no match

parseok: lea    di,[di+12]      ;12 = 8 + dot + ext
         lea    si,[di-1]       ;11 = 8 + ext (FCB)
         std                    ;------------------
         mov    cl,3            ;   $$example ext
         rep    movsb           ;to $$example eext
         mov    byte ptr [di],'.'
parsesp: lodsb                  ;to $$example .ext
         cmp    al,20h
         je     parsesp         ;get rid of padding
         cld                    ;------------------
         inc    si              ;last VCHAR in name
         inc    si              ;first space if any
         mov    cl,5            ;5 = dot + ext + nul
         xchg   di,si           ;   $$example .ext
         rep    movsb           ;to $$example.ext

         mov    si,offset comline
         cmp    byte ptr [si+1],':'
         pop    di              ;to .\example.ext (no drive)
         je     ccdrive         ;to X:example.ext (drive X:)
         mov    si,offset cwdpath
ccdrive: movsw

         mov    al,'.'
         lea    si,[di+15]      ;COM or EXE (initialized data)
         mov    cl,9
         repne  scasb           ;skip file name and guaranteed '.'
         mov    cl,3
         repe   cmpsb           ;compare upper case parse result
parserr: pop    cx
         pop    ax
         ret                    ;NE: no match, E: COM or EXE
comorexe endp

;-----------------------------------------------------------------------
initpath proc   near            ;find PATH in environment ES
         mov    ax,3700h        ;while at it get switchar
         mov    dx,'//'
         int    21h
         or     al,al           ;get DOS switchar DL '-'
         jnz    getenv          ;if 3700h unsupported
         cmp    dl,dh           ;if switchar default '/'
         je     getenv          ;keep path separator '\'
         mov    cwdpath+1,dh    ;else path separator '/'

getenv:  xor    di,di           ;ES:DI begin environment
         xor    ax,ax

getpath: mov    si,offset pathtxt
         mov    cx,5            ;length 'PATH=' at DS:SI
         repe   cmpsb
         je     stopath
         mov    cx,di           ;terminate search at segment limit
         not    cx              ;this will work for length < 64 KB

         repne  scasb           ;look for terminating NUL byte
         jcxz   nulpath         ;oops, DI = 0 wrap around at 64 KB
         cmp    es:[di],al
         jne    getpath         ;continue search unless double NUL

nulpath: mov    di,offset pathtxt+5
         push   ds
         pop    es              ;ES:DI far ptr empty ASCIIZ string

stopath: mov    word ptr pathptr+0,di
         mov    word ptr pathptr+2,es
         ret                    ;AX 0, DL switchar, ES undefined
initpath endp

;-----------------------------------------------------------------------
nextpath proc   near            ;search for path in environment
         push   ds              ;modifies AX, DI, SI
         mov    ah,-1           ;AH -1: yet no characters
         mov    di,offset command
         lds    si,pathptr
         assume ds:nothing
         test   ah,[si]
         jnz    incpath         ;handle ASCIIZ string end

         pop    ds              ;restore own data segment
         stc                    ;indicate PATH end (carry)
         ret

incpath: lodsb
         cmp    al,';'
         je     textend         ;handle PATH token end ';'
         or     al,al
         jz     textend         ;handle ASCIIZ string end
         stosb
         mov    ah,0            ;AH 0: token is not empty
         jmp    incpath

textend: pop    ds
         assume ds:DGROUP
         mov    word ptr pathptr,si
         or     ah,ah
         jnz    nextpath        ;AH not 0: token is empty

         cmp    byte ptr [di-1],':'
         jz     pathend         ;':' or DOS path separator
         mov    al,cwdpath+1
         cmp    [di-1],al
         jz     pathend         ;otherwise add '\' or '/'
         stosb

pathend: clc                    ;indicate token (no carry)
         ret                    ;DI points to end of token
nextpath endp

;-----------------------------------------------------------------------
doithere proc   near            ;try path\name.com or path\name.exe
         mov    si,2            ;skips .\ or drive ?: in com or exe
         jmp    short addpath
doit_cwd label  near
         xor    si,si           ;keeps .\ or drive ?: in com or exe
addpath: push   di
         push   si
         add    si,offset com
         call   near chkhere    ;try xxxxxx.COM at DI
         pop    si
         pop    di
         jnc    tryhere         ;NC: found, C: try EXE
         add    si,offset exe
         call   near chkhere    ;try xxxxxx.EXE at DI
tryhere: ret                    ;NC: found, C: not here

chkhere: mov    dx,si           ;add COM or EXE at DI
         call   strlen
         inc    cx
         rep    movsb
         mov    dx,bx           ;command file path
         mov    ax,4300h        ;for path at DS:DX
         int    21h             ;get file attribute
         retn                   ;NC: found, C: not here
doithere endp

;-----------------------------------------------------------------------
;        execute DOS external command, common exit handling

NEWSTK   equ    168             ;the stack for newdos
SDAPTR   equ    NEWSTK+168      ;new stack for exec, offset _BSS copy

;CAVEAT: main + SDAPTR is used as far pointer (overwriting pathptr copy)

exec     proc   near
         mov    si,offset _edata
         mov    di,offset main + SDAPTR
         mov    cx,si
         sub    cx,di           ;fixup for moved _BSS
         mov    bx,offset execpar
         mov    ax,[bx+2]
         sub    ax,cx           ;CS offset arguments ptr.  (comline)
         mov    [bx+2],ax
         mov    [bx+4],cs       ;PSP:fcb1/2 far ptr in execpar okay
         sub    bx,cx           ;CS offset exec parameters (execpar)
         sub    dx,cx           ;CS offset program path    (command)

        ifndef  __TINY__        ;---------------------------------------
         push   cs
         pop    es              ;ES = CS
         push   cs
         pop    ss              ;SS = CS    \  / keep this
        endif                   ;------------><-------------------------
         mov    sp,di           ;move stack /  \ together
         assume es:_TEXT

         mov    cx,offset _end
         sub    cx,si           ;CX = size of _BSS
         rep    movsb           ;move _BSS to begin of code
         push   bx              ;CS:BX exec parameters
         push   dx              ;CS:DX program path

        ifndef  __TINY__        ;-------------- EXE code + PSP ---------
         mov    es,word ptr fcb1+2
         mov    bx,offset endcode+15+100h
        else                    ;-------------- COM code + PSP ---------
         mov    bx,offset endcode+15
        endif                   ;---------------------------------------
         shr    bx,4            ;BX = paragraphs needed
         mov    ah,4Ah          ;4Ah: reallocate memory ES
         int    21h             ;free _DATA, _BSS, STACK
         jc     execerr         ;report unexpected error
         mov    ax,5D06h        ;---------------------------
         int    21h             ;get DOS SDA (DS:SI), DOS 3.x and 5+
         jc     execerr         ;DOS 4.x SDA function 5D0Bh unsupported
         mov    ax,ds
         push   cs              ;let EXE access code as data
         pop    ds              ;check effect of assume here
         assume ds:_TEXT, es:nothing, ss:nothing
         mov    word ptr main+SDAPTR+0,si       ;\/ overwriting
         mov    word ptr main+SDAPTR+2,ax       ;/\ pathptr copy
         mov    word ptr pspdta+2,es
         mov    word ptr newstack+2,cs

         mov    ax,3521h        ;---------------------------
         int    21h             ;get current INT 21h address
         mov    si,offset newdos
         mov    dx,si           ;DX     = SI =  offset new
         mov    [si+2],bx       ;[SI+2] = BX =  offset old
         mov    [si+4],es       ;[SI+4] = ES = segment old
         mov    ah,25h
         int    21h             ;set new INT 21h interrupt
         push   cs              ;---------------------------
         pop    es              ;ES = DS = CS
         pop    dx              ;DS:DX program path
         pop    bx              ;ES:BX exec parameters
         mov    ax,4B00h        ;DOS exec function
         int    21h

         lahf                   ;AH = exec flags
         push   ax              ;AL = exec error (if C set)
         inc    byte ptr locked
         lds    dx,olddos
         assume ds:nothing
         mov    ax,2521h        ;restore old INT 21h address
         int    21h             ;---------------------------
         push   cs
         pop    ds
         assume ds:_TEXT
         pop    ax              ;AL = exec error (if C set)
         sahf                   ;AH = exec flags

         jc     execerr         ;Carry: exec did not work
         mov    ah,4Dh          ;get exit code from child
         int    21h
         or     ah,ah           ;AH 0: normal termination
         jz     exitall

execerr: mov    dx,offset execx2c
         xchg   ax,cx
         call   x2c             ;AH = flags for 4B00h error
         xchg   ax,cx
         mov    dx,offset execmsg
exitmsg: call   message
exitall: mov    ah,4Ch          ;terminating with return code AL
         int    21h

execmsg  db     7,13,10,'DOS error 0x'
execx2c  db     '1234',13,10,0  ;ASCIIZ in code (_DATA released)
exec     endp

;-----------------------------------------------------------------------
x2c      proc   near            ;CX to ASCII (result buffer DS:DX) +++++
         push   es
         push   di              ;keeps ES:DI and CX
         push   ax              ;keeps DS:DX and AX

         push   ds
         pop    es
         mov    di,dx           ;ES:DI result DS:DX
         mov    ax,4            ;AX = 4 hex. digits
         xchg   cx,ax           ;AX = CX, CX = 4

nibble:  rol    ax,4
         push   ax
         and    al,0Fh          ;++++++++ X2C credits to Tim Lopez +++++
         cmp    al,0Ah          ;0..9  C; -6A 96..9F C  A; -66 30..39
         sbb    al,69h          ;A..F NC; -69 A1..A6 C NA; -60 41..46
         das                    ;read NC = No Carry, C NA = C + No Aux.
         stosb                  ;ASCII result
         pop    ax
         loop   nibble

         xchg   cx,ax           ;restore CX
         pop    ax              ;restore AX
         pop    di              ;restore DI
         pop    es              ;restore ES (result pointer DS:DX) +++++
         ret
x2c      endp

;-----------------------------------------------------------------------
message  proc   near            ;ASCIIZ string DS:DX to STDERR
         push   ax              ;keeps DS:DX, AX
         call   strlen          ;modifies BX, CX
         mov    bx,2            ;stderr
         mov    ah,40h          ;output
         int    21h
         pop    ax
         ret
message  endp

;-----------------------------------------------------------------------
strlen   proc   near            ;return length ASCIIZ string DS:DX
         push   es              ;keeps ES:DI, DX
         push   di              ;modifies CX, AX

         push   ds
         pop    es
         mov    di,dx           ;string ES:DI
         mov    cx,di
         not    cx              ;terminate search at segment limit
         xor    al,al           ;this will work for length < 64 KB
         repne  scasb           ;search NUL
         mov    cx,di           ;DI points behind NUL (or is NULL)
         sub    cx,dx           ;length including NUL
         dec    cx              ;length excluding NUL

         pop    di              ;restore DI
         pop    es              ;restore ES
         ret                    ;returns CX string length, AL 0
strlen   endp

;-----------------------------------------------------------------------
KBMAGIC  equ    "KB",0,0EBh,0,0CBh  ;see AMIS specification
         align  4
         assume nothing
         assume cs:_TEXT

newdos   proc   far
         jmp    short chkdos    ;this is actually a "KB" structure...
olddos   dd     0               ;far address old interrupt 21h
         db     KBMAGIC         ;"KB" AMIS specification 3.6 compatible

pspdta   dd     DTA             ;PSP (= PID) segment fixup at runtime
newstack dd     offset main + NEWSTK
oldstack dd     0               ;stack switch in showbx
locked   db     -1              ;non-reentrant code lock
regio    db     13,10,">",0     ;show < on exit
reg32    db     ", ds ax ",0    ;set by showio
regcx    db     "1234",0        ;set by showcx
        ;xlatb: 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, A, B (function 00..0B)
chktab   db     0, 1, 3, 0, 0, 0, 1, 1, 1, 3, 1, 1

chkdos:  cmp    ah,0Ch
         ja     chkstd          ;DOS functions AH 0D..FFh
         jb     chkcon          ;DOS functions AH 00..0Bh
         call   near chksub     ;check 0Cxxh subfunctions:
         jpo    jmpdos          ;input AL 1, 6, 7, 8, A, B
         jpe    callos          ;other AL 0, 2, 3, 4, 5, 9

chkcon:  xchg   ah,al
         call   near chksub
         xchg   ah,al
         jz     callos          ;AH 0, 3, 4, 5 (not stdio)

jmpdos:  jmp    cs:[olddos]     ;original interrupt

chksub:  push   ax
         push   bx
         mov    bx,offset chktab
         xlat   cs:             ;table in code segment
         or     al,al           ;AL 0: ZR + PE
         pop    bx              ;AL 1: NZ + PO
         pop    ax              ;AL 3: NZ + PE
         retn                   ;near inline proc

chkstd:  cmp    ax,4406h        ;handle BX status input
         je     chk012
         cmp    ax,4407h        ;handle BX status output
         je     chk012
         cmp    ah,3Fh          ;handle BX input
         jb     callos
         cmp    ah,40h          ;handle BX output
         ja     callos
chk012:  cmp    bx,2            ;handle stdin / stdout / stderr
         jbe    jmpdos          ;handle BX 2 (stderr) used here

callos:  push   ds              ;store caller DS
         push   cs
         pop    ds
         assume ds:_TEXT
         mov    byte ptr regio+2,">"
         call   showio          ; > entry register values shown

         pop    ds              ;reset caller DS
         assume ds:nothing
         or     ah,ah           ;TBD:  AH 0 exit causes havoc
         jz     jmpdos          ; for a modified caller stack

         push   cs              ;10 own segment \/ for
         push   offset retpsp   ;+8 own  offset /\ RETF
         pushf                  ;+6 dummy flags \/ +16 IRET flags
         push   cs              ;+4 dummy (PSP) /\ +14 IRET segment
         push   52h             ;+2 offset RETF || +12 IRET offset
         push   bp              ;save caller BP
         mov    bp,sp           ;=0
         push   [bp+16]         ;caller IRET flags
         pop    [bp+06]         ;copied IRET flags
         push   ds              ;save caller DS
         mov    ds,[bp+14]      ;caller IRET CS
         mov    [bp+04],ds      ;copied IRET CS
         cmp    byte ptr ds:[bp+02],0CBh
         pop    ds              ;reset caller DS
         pop    bp              ;reset caller BP
         je     jmpdos          ;PSP:[52h] is RETF

         add    sp,4            ;===== plan B =====
         call   cs:[olddos]     ;emulate INT 21h
         retf                   ;own RETF to retpsp

retpsp:  push   bp              ;interrupts are likely enabled
         mov    bp,sp           ;BP+6: old flags (RETF 2)
         pushf
         pop    [bp+06]         ;BP+6: new flags (IRET)
         pop    bp

         push   ds              ;store result DS
         push   cs
         pop    ds
         assume ds:_TEXT
         cli                    ;interrupt state reset by IRET
         mov    byte ptr regio+2,"<"
         call   showio          ; < exit register values shown

         pop    ds              ;reset result DS
         iret                   ;reset result flags
newdos   endp

;-----------------------------------------------------------------------
showio   proc   near            ;retn BP+8      ;use copied result flags
         push   dx              ; dx: BP+6      ; fl: BP+16 result/entry
         push   cx              ; cx: BP+4      ; cs: BP+14
         push   bx              ; bx: BP+2      ; ip: BP+12
         push   bp              ; bp: BP+0      ; ds: BP+10
         mov    bp,sp
         inc    byte ptr locked
         jnz    unlock          ;.386 OP: 0F 85 (15bit offset)

         mov    bx,offset reg32+5
         mov    dx,offset regio
         call   showdx
         mov    word ptr [bx],"xa"
         mov    cx,ax           ; AX
         call   showcx
         mov    word ptr [bx],"xc"
         mov    cx,[bp+4]       ; CX
         call   showcx
         mov    word ptr [bx],"sd"
         mov    cx,[bp+10]      ; DS
         call   showcx
         mov    byte ptr [bx-3],"d"
         push   es
         mov    es,[bp+10]      ; DS for DS:DX
         mov    word ptr [bx],"xd"
         mov    cx,[bp+6]       ; DX
         call   showbx          ; ds:[dx+0], ds:[dx+2], etc.
         pop    es

         mov    word ptr [bx],"sc"
         mov    cx,[bp+14]      ; CS
         call   showcx
         mov    word ptr [bx],"pi"
         mov    cx,[bp+12]      ; IP
         call   showcx
         mov    word ptr [bx],"se"
         mov    cx,es           ; ES
         call   showcx
         mov    byte ptr [bx-3],"e"
         mov    word ptr [bx],"xb"
         mov    cx,[bp+2]       ; BX
         call   showbx          ; es:[bx+0], es:[bx+2], etc.

         mov    word ptr [bx],"ss"
         mov    cx,ss           ; SS
         call   showcx
         mov    word ptr [bx],"ps"
         lea    cx,[bp+12]      ; SP
         call   showcx
         mov    word ptr [bx],"sf"
         mov    cx,fs           ; FS
         call   showcx
         mov    byte ptr [bx-3],"d"
         push   es
         mov    es,[bp+10]      ; DS for DS:SI
         mov    word ptr [bx],"is"
         mov    cx,si           ; SI
         call   showbx          ; ds:[si+0], ds:[si+2], etc.
         pop    es

         mov    cx,"<"
         xchg   byte ptr regio+2,ch
         cmp    ch,cl
         mov    word ptr [bx],"lf"
         mov    cx,[bp+16]      ; flags
         jne    showfl          ; on entry ignore carry
         test   cl,1            ; carry flag
         jz     showfl
         mov    word ptr [bx],"YC"
showfl:  call   showcx
         mov    word ptr [bx],"pb"
         mov    cx,[bp+0]       ; BP
         call   showcx
         mov    word ptr [bx],"sg"
         mov    cx,gs           ; GS
         call   showcx
         mov    byte ptr [bx-3],"e"
         mov    word ptr [bx],"id"
         mov    cx,di           ; DI
         call   showbx          ; es:[di+0], es:[di+2], etc.

unlock:  dec    byte ptr locked
         pop    bp
         pop    bx
         pop    cx
         pop    dx
         ret
showio   endp

;-----------------------------------------------------------------------
showbx   proc   near
         call   showcx          ;CX is caller's BX, DI, SI, or DX
         mov    dx,":"
         mov    byte ptr [bx-1],dl
         mov    byte ptr [bx+2],dh
         mov    dx,offset reg32
         call   showdx
         mov    dx,20h
         mov    word ptr [bx-1],dx
         mov    byte ptr [bx+2],dl

         push   bx
         inc    cx
         jcxz   segvbx          ;cannot access [BX] = [-1]
         dec    cx
         mov    bx,cx           ;ES:BX is ES:BX, ES:DI, DS:DX, or DS:SI
         mov    cx,es:[bx]
         call   showcx
         mov    dx,6

nextbx:  add    bx,2
         jc     segvbx          ;64 KB wrap around
         mov    cx,es:[bx]
         call   showcx
         dec    dx
         jnz    nextbx

segvbx:  pop    bx
         mov    dx,offset regio
         call   showdx

        ;----------------------- DTA buffer to stderr ------------------
         mov    word ptr pspdta,DTA
         mov    word ptr oldstack+0,sp
         mov    word ptr oldstack+2,ss
         lss    sp,newstack     ;.386 OP: 0F B2
        ifndef  __TINY__        ;-------------- EXE code != PSP --------
         push   ds
        endif                   ;---------------------------------------
         pusha

         lds    si,main+SDAPTR  ;DS:SI is DOS SDA
         assume ds:nothing
         cmp    word ptr [si],0 ;NZ: DOS busy, use int 29h
         lds    si,cs:pspdta
         assume ds:PSP
         je     showit

show29:  lodsb                  ;next ASCIIZ character
         or     al,al
         jz     shower
         int    29h             ;"fast putchar" AL to CON
         jmp    show29

showit:  mov    dx,si           ;DS:DX = PSP DTA
         sti                    ;(try to) be nice...
         call   message         ;clobbers BX, CX
         cli                    ;...protect oldstack

shower:  popa
        ifndef  __TINY__        ;-------------- EXE code != PSP --------
         pop    ds
        endif                   ;---------------------------------------
         assume ds:_TEXT
         lss    sp,oldstack     ;.386 OP: 0F B2
         ret
showbx   endp

;-----------------------------------------------------------------------
showcx   proc   near            ;convert CX to four hex. digits at DS:DX
         push   dx              ;no registers modified
         mov    dx,offset reg32+4
         call   showdx
         mov    dx,offset regcx
         call   x2c
         call   showdx
         pop    dx
         ret
showcx   endp

;-----------------------------------------------------------------------
showdx   proc   near            ;ASCIIZ string DS:DX appended to DTA
         push   es              ;modifies DTA and offset (pspdta+0)
         push   di
         push   si              ;keeps ES:DI, DS:SI, DX, AX, CX
         push   ax
         push   cx              ;avoid PUSHA on dubious caller stack

         call   strlen          ;AL = 0, CX = length
         les    di,pspdta       ;ES:DI offset in DTA
         mov    si,dx           ;DS:SI = DS:DX
         rep    movsb           ;ES:DI + CX characters
         mov    word ptr pspdta,di
         stosb                  ;AL = NUL terminator

         pop    cx
         pop    ax
         pop    si
         pop    di
         pop    es
         ret
showdx   endp

;-----------------------------------------------------------------------
endcode  label  byte
         public main            ;public symbols for WDISasm listing
         public initpath
         public nextpath
         public comorexe
         public doithere
         public doit_cwd
         public exec            ;can overwrite main .. doit_cwd
         public x2c             ;used by exec and showcx
         public message         ;used by exec and showbx
         public strlen          ;used by message

         public newdos
         public olddos
         public newstack
         public oldstack
         public pspdta
         public locked
         public showio
         public showbx
         public showcx
         public showdx
         public endcode

         public cwdpath         ;CWD '.\' (or './'), +1 path separator
         public exe             ;parse .\*.EXE or ?:*.EXE \/ current or
         public com             ;parse .\*.COM or ?:*.COM /\ given drive

         public pathptr         ;PATH value in environment segment
         public comline         ;arg.s word 2 etc. (command tail)
         public command         ;arg.s word 1 expanded to COM/EXE
         public execpar         ;+2 far ptr comline, +6 far ptr PSP:fcb1

;-----------------------------------------------------------------------
_TEXT    ends                               ;;; end of _TEXT segment ;;;
_DATA    segment use16 para public 'DATA'   ;;;  begin  DATA segment ;;;

cwdpath  db     '.','\'         ;CWD '.\', path separator '/' also works
pathtxt  db     'PATH=',0       ;search PATH setting

exe      db     '$$????????EXE',0,0,0,0,'EXE'   ;4 nulls for parse FCB
com      db     '$$????????COM',0,0,0,0,'COM'   ;EXE or COM extensions

crlf     db     13,10,0
missing  db     ' cannot run ',0
misscom  db     ' or ',0
usage2   db     ' command [arguments]',0

usage1   db     13,10,'usage: [LH] '    ;\/
dostool  db     'DOSMON',0              ;/\
         even
_DATA    ends                               ;;; end of  DATA segment ;;;

;-----------------------------------------------------------------------
STRINGS  segment use16 word public 'DATA'   ;;; DGROUP data segments ;;;
STRINGS  ends
CONST    segment use16 word public 'DATA'   ;;; DGROUP data segments ;;;
CONST    ends
DATA     segment use16 word public 'DATA'   ;;; DGROUP data segments ;;;
DATA     ends

;-----------------------------------------------------------------------
_BSS     segment use16 word public 'BSS'    ;;;  begin  _BSS segment ;;;

pathptr  dd     1 dup (?)       ;address actual PATH at begin (see exec)
comline  db     128 dup (?)     ;DTA copy used for DOS external command
command  db     128 dup (?)     ;buffer to construct PATH token command

execpar  dw     1 dup (?)       ;segment environment \  / structure
         dd     1 dup (?)       ;pointer command line \/  used for
fcb1     dd     1 dup (?)       ;pointer own 1st FCB  /\  DOS EXEC
fcb2     dd     1 dup (?)       ;pointer own 2nd FCB /  \ function

_BSS     ends                               ;;; end of  _BSS segment ;;;
;-----------------------------------------------------------------------
        ifndef  __TINY__        ;-------------- EXE segments: ----------
STACK    segment use16 para stack 'STACK'   ;;;  begin STACK segment ;;;
         db     512 dup (?)     ;size ignored by WLINK (minimal 512)
STACK    ends                               ;;; end of STACK segment ;;;
FAR_DATA segment use16 byte public 'FAR_DATA'
FAR_DATA ends                   ;establish use16 for dummy FAR_DATA
        .286                    ;force use16 segment (for WASM -3 etc.):
        .code                   ;establish use16 for dummy filename_TEXT
        endif                   ;---------------------------------------
         end    main            ;program entry point
