;-----------------------------------------------------------------------
; WASM (Open Watcom)
;       wasm -wx -we -q -3 -mt memdrive
;       wdis memdrive -l -fi
;       wlink format dos com file memdrive name memdrive.sys
; MASM (untested, might require an empty stack segment):
;       masm memdrive
;       link memdrive
;       exe2bin memdrive memdrive.sys
; Installation:
;       device[high]=[path\]memdrive.sys
; Requirements:
;       The resident part of MEMDRIVE needs 2208 bytes conventional
;       memory for 1.44 MB floppy images.  If XMS is available the
;       virtual floppy unsurprisingly needs 1440 KB XMS.  MEMDRIVE
;       requires an 80386 CPU and PC DOS 5 or better.

; Usage:
;       Without XMS MEMDRIVE creates one drive containing a single
;       file MEMORY.DAT corresponding to the memory below 1 MB.  Two
;       other directory entries HMA.DAT and DISK1440.IMG are deleted.
;       If HIMEM.SYS is installed 65520 bytes in HMA.DAT are visible.
;       If the free XMS space allows it MEMDRIVE creates a drive for
;       DISK1440.IMG, an initially empty formatted 1.44 data floppy.
;       All floppy access methods on the second drive should work as
;       expected, e.g., FORMAT, FORMAT /T:80 /N:18, FORMAT /F:1.44,
;       DISKCOMP, DISKCOPY, CHKDSK, Norton DISKEDIT, NDD, etc.

;       A DISKCOPY of the initial image to a real floppy results in a
;       non-bootable floppy returning control to BIOS INT 18h without
;       user intervention.

;       DISK1440.IMG in the first drive can be deleted to work with
;       other images, just "insert" (i.e., copy) another floppy image
;       in the first drive.  The content is immediately available as
;       virtual floppy in the second drive.

; Features:
;       The four system sectors (MBR, FAT, FAT copy, and root directory)
;       on the first drive are to varying degrees virtual, e.g., unused
;       zero bytes do not really exist in memory.  The directory entries
;       for MEMORY.DAT and HMA.DAT cannot be modified.  The volume label
;       directory entry can be removed for experiments with two writable
;       directory entries instead of one.

;       To get a dump of the memory below 1 MB simply copy MEMORY.DAT.
;       Low level write access (INT 26h) on the MEMORY.DAT and HMA.DAT
;       sectors is possible and expected to crash the system.  SCANDISK
;       or other disk surface tests on the first drive will cause havoc,
;       but NDD 8.0 is okay.

;       The file date of MEMORY.DAT is the BIOS date.  The date of the
;       volume label is the date of the MEMDISK version, time 12:34:56.
;       Other timestamps correspond to the installation (e.g., reboot)
;       time.  The volume serial number for the 2nd drive is shown by
;       VOL and DIR as ccyy-mmdd (installation date BCD ddmmyycc).  For
;       the 1st drive it is ddhh-mmss (installation time BCD ssmmhhdd).

;       Formats requiring more than 1440 KB (e.g., 2M, FDREAD, or XDF)
;       cannot work.  Sector interleaves or checksums are not emulated.
;       A DISKCOPY from non-standard floppies should fail; check it out.

;       The MEMDRIVE source is prepared to create a 2880 KB variant by
;       defining the symbol EXTRA.  After that DISK2880.IMG works like
;       DISK1440.IMG.  The resident code for the 2880 KB variant needs
;       64 bytes more for 1440/32*(3/2) FAT12 bytes on the first drive.
;       The source also supports a FAT16 2.88 MB variant with CLUSTER=1
;       instead of CLUSTER=2 (see below), but PC DOS 7 FORMAT /F:2.88
;       expects FAT12.

;       The virtual floppy is write-protected while the read-only bit
;       in the attributes for the disk image file is set.  This bit is
;       automatically set or cleared if DOS sets or clears the access
;       flag for the drive, e.g., if the disk image is not formatted.
;       Use ATTRIB or EJECT to modify the disk image file attributes.

;       The virtual floppy drive door is locked while the system bit in
;       the attributes for the disk image file is set.  This bit is set
;       or cleared for the relevant CD-ROM IOCTL used by DRVLOCK.EXE.
;       If the drive door is locked the virtual cannot be ejected with
;       a related CD-ROM IOCTL.

;       If permitted EJECT.EXE resets any logical volume lock, the open
;       counter, and the disk image attributes.  Generic IOCTL 849h has
;       the same effect (untested MS DOS 7 feature).

;       All files "inserted" in the first drive should have the exact
;       size of a floppy disk image (1474560 bytes for a 1.44M floppy).
;       This allows DOS to use the free clusters on the first drive in
;       a predictable way.  If a shorter file is "inserted" and later
;       replaced by another disk image DOS allocates clusters beginning
;       after the end of the replaced shorter file, and what should be
;       the boot sector, FAT, and root directory of the virtual floppy
;       is random garbage.  After a reboot DOS and MEMDRIVE are again
;       automatically synchronized.  For what it's worth a smaller disk
;       image works as expected (tested with a 720 KB STACKER image),
;       but after that the insertion of another disk image file fails.

; PC DOS 7
;       PC DOS 7 adds 3 paragraphs (48 bytes) to the resident code per
;       subunit for its own purposes.  The last 15 bytes are apparently
;       not used.  The PC DOS 7 init request structure length can be 24,
;       i.e., the init error flag (cf. INIERR) flag is a byte instead
;       of a word.

; Free DOS:
;       FreeDOS FORMAT.EXE 0.91v misinterprets PC DOS 7 as MS DOS 7 and
;       then uses IOCTL 84Ah/86Ah to lock/unlock a logical volume.  It
;       apparently uses DOS function 4409h ("is drive remote") instead
;       of 4408h ("is drive removable") to identify removable drives.
;       If the format confirmation is not "YES" it forgets to unlock the
;       drive, otherwise it tries to lock the drive again.  Maybe use
;       SETVER.EXE, DOS622.COM, or TV-LIE.COM (the "true version lie"
;       TSR shipped by IBM) to fake the DOS version for this FORMAT.EXE.

; Development:
;       The boot sector is almost used up to the last byte.  Be careful
;       with the size of the "overlay" part of the code at the end after
;       procedure "format".  The initialization code removes itself when
;       done.  You can add new resident code near the begin (before the
;       boot sector).

;       At some places 80386 instructions are used.  This requires WASM
;       option -3 and explicit USE16 segments, because the -3 default is
;       USE32.  At some places the code abuses the fact that subunit 0
;       has one head and subunit 1 two heads/sides.

;       IOCTL 861h/841h (track i/o) and 866h/846h (get/set volume label)
;       work as expected.  860h/840h (get/set parameters), 868h (sense
;       media), 862h/842h (verify/format track), and 867h/847h (get/set
;       flag) are mostly unclear, but apparently good enough for FORMAT.

;       IOCTL 847h (set access flag, only subunit 1) sets or clears the
;       read-only attribute of the disk image file.  Maybe this helps to
;       figure out who uses this function.  IOCTL 869h is implemented as
;       NOP, because MEMDRIVE supports generic IOCTLs as 86?h/84?h pairs.

;       IOCTL 848h (media lock) manages a lock counter per subunit.  The
;       counter can be queried, incremented, and decremented.  Underflow
;       and overflow are reported as "busy".  Likewise the generic IOCTL
;       check (driver function 19h) reports an unsupported generic IOCTL
;       as "busy" and not as error.  The specific IOCTL read and write
;       functions (used by PC DOS 7 DRVLOCK + EJECT) also report "busy"
;       for unexpected i/o requests, e.g., subunit 0 has no door lock
;       and cannot be ejected.

;       IOCTL 84Ah/86Ah (lock/unlock logical drive, only subunit 1) are
;       implemented as lock count 0 to 255 and 255 to 0, respectively.
;       The drive cannot be locked for format if its lock count was not
;       zero, and the counter cannot be incremented while it is already
;       locked for format.  A misbehaving application could decrement
;       the lock counter in this state resulting in spurious errors.

;                                               (2010, Frank Ellermann)
;-----------------------------------------------------------------------

.errndef __TINY__               ;DOS device drivers are tiny (DS == CS)
        .386                    ;allow .386 instructions (MOVSD, etc.)
DGROUP   group  _TEXT           ;put everything in one _TEXT segment
_TEXT    segment use16 para public 'CODE'
         assume nothing
         assume cs:DGROUP

device   dd     -1              ;device link
attrib   dw     48C0h           ;device attribute 0100 1000 1100 0000
         dw     offset strategy ;device strategy  entry offset
         dw     offset intentry ;device interrupt entry offset
         db     2               ;subunits, followed by 7 nulls
         db     7 dup (0)       ;          (instead of a name)

; 0100 1000 1100 0000
; FEDC BA98 7654 3210 attribute bits for block devices (DOS 5+)
; |||| |    ||     |
; |||| |    ||     +- n/a: 32bit sector numbers (BIGFAT, FAT32)
; |||| |    |+------- SET: generic IOCTL (13h, implies 17h + 18h)
; |||| |    +-------- SET: support IOCTL check (function 19h)
; |||| +------------- SET: OCR (open 0Dh, close 0Eh, removable 0Fh)
; |||+--------------- n/a: "remote" device (cf. INT 21h AX 4409h)
; ||+---------------- n/a: "non-IBM" format affects function 02h
; |+----------------- SET: specific IOCTL (functions 03h and 0Ch)
; +------------------ n/a: device type (0: block, 1: character)

;-----------------------------------------------------------------------
; definition of request offsets and magic numbers:

CMD      equ    02              ;offset command
STATUS   equ    03              ;offset status code
MEDIA    equ    13              ;offset media desc. / supported drives
GENF     equ    13              ;offset generic IOCTL major/minor func.
BUFFER   equ    14              ;offset I/O buffer / device end address
COUNT    equ    18              ;offset I/O buffer length bytes/sectors
CONFIG   equ    18              ;offset addr. init. string [BPB pointer]
GENP     equ    19              ;offset generic IOCTL parameter address
FIRST    equ    20              ;offset start sector short
VOLID    equ    22              ;offset drive number/volume ID address
INIERR   equ    23              ;offset addr. init. error flag (DOS 5+)
FIRST32  equ    26              ;offset start sector long (attr. bit 1)

DELETED  equ    0E5h            ;for deleted directory entries if no XMS
MAGIC    equ    0AA55h
LOCKED   equ    00000100b       ;system attribute bit emulates door lock
MEDIA1   equ    248             ;media descriptor F8h (fixed disk)
MEDIA2   equ    240             ;media descriptor F0h (1440/2880)
LENSEC   equ    512             ;512 / 16 = 32 paragraphs per sector
PARSEC   equ    5               ;  2 ** 5 = 32 paragraphs per sector
SYSSEC   equ    4               ;MBR + 1st FAT + 2nd FAT + root dir.
LOWSEC   equ    2048            ;1 MB = 2048 * 512 bytes
HMASEC   equ    128             ;64 KB = 128 * 512 bytes (actually -16)

FIXTRACK equ    00000100b       ;bit 2 immutable track layout
DOORLOCK equ    00000010b       ;bit 1 changeline supported
        ;try to set bit 3 in DOORLOCK and bit 0 in FIXTRACK (unclear)

        ifndef  EXTRA           ;HD  (high density) 1440 KB image:
MULSEC   equ    44              ;5060 = 44 * 115 tracks on subunit 0
DISKTYPE equ    7               ;7 1440, 6 tape, 5 fixed, 2 720
DISKNAME equ    "DISK1440"
XMSSEC   equ    18*2*80         ;2880 * 512 bytes for 1440 KB
CLUSTER  equ    1               ; 1 sector  per cluster (for F9h 2)
FATSEC   equ    9               ; 9 sectors per FAT     (for F9h 3)
DIRSEC   equ    13+CLUSTER      ;14 sectors root dir.   (for F9h 7)
        else                    ;ED (extra density) 2880 KB image:
MULSEC   equ    20              ;7940 = 20 * 397 tracks on subunit 0
DISKTYPE equ    9               ;9 2880, 8 optical, 7 1440, 5 fixed
DISKNAME equ    "DISK2880"
XMSSEC   equ    36*2*80         ;5760 * 512 bytes for 2880 KB
CLUSTER  equ    2               ; 2 sectors per cluster (or try 1)
        if      CLUSTER eq 2    ; 2 is compatible with FORMAT /2.88
FATSEC   equ    9               ; 9 sectors per FAT
DIRSEC   equ    13+CLUSTER      ;15 sectors root dir. for 240 entries
        endif
        if      CLUSTER eq 1    ;CAVEAT: DISKTYPE 9 might be not okay
FATSEC   equ    23              ;23 sectors per FAT
DIRSEC   equ    13+CLUSTER      ;14 sectors root dir. for 224 entries
        endif
        endif
MEMSEC   equ    SYSSEC+LOWSEC+HMASEC+XMSSEC

; Sanity checks: the TOTAL number of clusters should cover all remaining
; sectors for the chosen DIRSEC and FATSEC values.  TOTAL has to be less
; than 4086 (0FF6h) for FAT12, because cluster numbers start at 2, i.e.,
; -10: last cluster, bad: -9, EOF: -8..-1, free: 0, unused: 1, first: 2.

TOTAL    =      (XMSSEC-(1+2*FATSEC+DIRSEC)+(CLUSTER-1))/CLUSTER
.errnz          (XMSSEC-(1+2*FATSEC+DIRSEC)-(CLUSTER*TOTAL))
.errnz          XMSSEC gt 65535 ;BIGFAT not implemented
.errnz          0FFF6h le TOTAL ; FAT32 not implemented
        if      00FF6h gt TOTAL ; FAT12:
FATNAME  equ    "FAT12   "      ;DOS evaluates TOTAL and ignores FATNAME
FATFOUR  equ    0               ;fourth byte of FAT12 for initial format
        else                    ; FAT16:
FATNAME  equ    "FAT16   "      ;typically only used for at least 32 MB
FATFOUR  equ    255             ;fourth byte of FAT16 for initial format
        endif

; The predefined FAT for the control unit 0 uses 32 KB clusters for a
; small resident code size.  Putting 720 KB images in the 1440 KB slot
; works - also tested with a STACKER image.  The remaining 720 KB are
; intentionally unavailable, because "inserting" disk images only works
; at the begin of the XMS memory.  If necessary the volume label can be
; deleted to make room for a subdirectory with ordinary RAM disk files.

; In theory the maximal FAT16 cluster number 65526 (0FFF6h) allows
; CLUSTER 1 for 2880 KB.  However, PC DOS 7 FORMAT option /F:2.88 does
; not work as expected for FAT16.  FORMAT without option or options
; /T:80 /N:36 is okay.

;-----------------------------------------------------------------------
; Serialization strategy:  Each request is initialized with status busy.
; If the old request offset is -1 (invalid) a new request can be noted.
; The request offset is invalidated (-1) at the begin of intentry.  If
; the request offset is invalid intentry exits.  A "running" semaphore
; is incremented at the start of intentry and decremented at exit.  The
; initial running value is -1 and permits intentry to process a request.

strategy proc   far             ;store request block address
         pushf
         cli
         mov    word ptr es:[bx+STATUS],200h    ;status busy, not ready
         cmp    word ptr cs:request+0,-1        ;at most one request
         jne    intbusy
         mov    word ptr cs:request+0,bx        ;ES:BX is address...
         mov    word ptr cs:request+2,es        ;...of request block
intbusy: popf
         retf
strategy endp

;-----------------------------------------------------------------------
intentry proc   far             ;execute request block
         pushf
         pusha                  ;save registers
         push   ds
         push   es
         cld
         push   cs
         pop    ds              ;address data in code segment
         assume ds:DGROUP

         inc    byte ptr running
         jnz    intnop
         les    di,request      ;request ES:DI
         mov    si,-1           ;invalidate request
         cmp    si,di
         je     intnop          ;no pending request
         mov    word ptr request,si

         mov    ax,8001h        ;status unknown subunit
         mov    dx,es:[di+CMD-1]
         cmp    dl,2            ;unit DL = es:[di+CMD-1]
         jnb    interr
         or     dl,dl
         jz     intown          ;subunit 0 needs no XMS
         cmp    word ptr xmsdata,si
         je     interr          ;uninitialized subunit 1

intown:  mov    al,dh           ;AL = DH = es:[di+CMD]
         cbw                    ;CMD byte AL to word AX
         mov    bx,8003h        ;AX = CMD, BX = status
         cmp    al,UNKNOWN      ;test CMD function number
         xchg   ax,bx           ;BX = CMD, AX = status
         jnb    interr          ;AX = 8003h unknown function

         mov    word ptr oldstack+0,sp
         mov    word ptr oldstack+2,ss
         add    bx,bx           ;BX = offset of CMD function
         jz     intcmd          ;Z: stack of caller for init
         lss    sp,newstack     ;NZ: init code used as stack
         sti

intcmd:  xor    bp,bp           ;call command with BP = 0
         xor    ax,ax           ;call command with AX = 0 ok
         mov    dh,al           ;call command for subunit DX
         push   es
         push   di
         call   [commands+bx]
         assume ds:nothing      ;commands may modify DS
         pop    di
         pop    es              ;request ES:DI
         push   cs
         pop    ds              ;address data in code segment
         assume ds:DGROUP
         cli
         lss    sp,oldstack     ;restore stack of caller

interr:  or     ah,1            ;status ready bit
         mov    es:[di+STATUS],ax
intnop:  dec    byte ptr running
         pop    es              ;restore registers
         pop    ds
         popa
         popf
xmsdummy label  far             ;used if HIMEM.SYS unavailable
         retf
intentry endp

;-----------------------------------------------------------------------
; device function 0Fh (removable media query, OCR attribute bit)

         assume ds:DGROUP, ss:DGROUP

fixed    proc   near            ;AX 0, DX subunit, ES:DI request
         or     dx,dx           ;subunit DX
         jnz    notbusy         ;subunit 1 is removable
setbusy: mov    ah,2            ;subunit 0 is a fixed disk
notbusy: ret                    ;"busy bit" indicates the result
fixed    endp

;-----------------------------------------------------------------------
; device functions 17h/18h get/set logical device (DRIVER.SYS support):
; If "set" does nothing PC DOS 7 DISKCOPY and DISKCOMP fail.  The same
; code for "set" and "get" is apparently good enough.

getset   proc   near            ;AX 0, DX subunit, ES:DI request
         mov    es:[di+1],al    ;logical device cannot be switched
         ret
getset   endp

;-----------------------------------------------------------------------
; device function 19h (generic IOCTL ceck, device attribute bit 7)

gentest  proc   near            ;AX 0, DX subunit, ES:DI request
         mov    cl,GENF+1       ;expect IOCTL word ES:[DI+GENF]
         call   chkbuf
         call   genlook         ;generic IOCTL 840..84A, 860..86A check
         jnz    setbusy         ;unsupported reported via "busy bit"
         ret
gentest  endp

;-----------------------------------------------------------------------
; device functions 03h/0Ch (IOCTL read/write)
; Raw mode (ECC) and audio CD-ROM IOCTLs are not applicable.  Most other
; CD-ROM IOCTLs are designed for a character device and pointless for a
; block device.  EJECT + (UN)LOCK DOOR are supported for subunit 1.  The
; status and capabilities with the door lock bit can be queried.
; The driver header address and the volume size can be also determined
; with CD-ROM IOCTLs.  To simplify the code RESET is not yet supported,
; as it would be an unclear subset of EJECT.  Define COOKED to get rid
; of the code for a cooked mode sector size query.

ioctlrd  proc   near            ;AX 0, DX subunit, ES:DI request
         inc    bp              ;BP 1 (read)
ioctlwr  label  near            ;BP 0 (write) set by intentry
         mov    cl,COUNT+1      ;expect word at ES:[DI+COUNT]
         call   chkbuf
         mov    cx,es:[di+COUNT]
         jcxz   notbusy         ;length 0: simply do nothing
         or     dx,dx
         jz     setbusy         ;subunit 0 not supported
         call   lesdibuf        ;ES:DI = caller buffer
         mov    bx,es:[di]      ;get first word in i/o buffer
         mov    si,offset dskentry+11
         mov    dh,[si]         ;DH = disk image attributes
         or     bp,bp
         jnz    ioctlin         ;IOCTL read

         cmp    bl,dl           ;expect BL <= 1 for PC DOS 7
         jnbe   setbusy         ;BL 2 (reset) not yet supported
         jne    eject           ;BL 0: expect AX 0, DX 1, BP 0
         or     dh,LOCKED
         cmp    bh,bl           ;BL 1: expect BH <= 1
         jnbe   setbusy
         je     setdoor
         xor    dh,LOCKED
setdoor: mov    [si],dh         ;door (un)locked
         ret

ioctlin  label  near
        ifndef  COOKED          ;---------------------------------------
         cmp    bx,7            ;implemented: BX 7 or BL 0, 6, 8
         je     ioctl_7         ;(raw mode BX 0107h not supported)
        endif                   ;---------------------------------------
         cmp    cx,5
         jb     setbusy         ;5 = 1 + dword required for BL 0, 6, 8
         xor    cx,cx
         cmp    bl,ch           ;0 (get driver header for MSCDEX)
         jne    ioctl_6
         mov    si,cs           ;SI:CX = CS:0
         jmp    short ioctlnn

ioctl_6: cmp    bl,6            ;6 (query status + capabilities)
         jne    ioctl_8
         xor    si,si           ;capabilities (bits 2..31) clear
         test   dh,LOCKED       ;hint, emulated door is never open
         jnz    ioctlnn
         mov    cl,2            ;SI:CX = 2 (unlocked) or 0 (locked)
         jmp    ioctlnn
        ifndef  COOKED          ;---------------------------------------
ioctl_7: cmp    cx,4            ;7 (BH 0: cooked mode sector size)
         jb     setbusy         ;4 = 2 + word required for BL 7
         mov    word ptr es:[di+2],LENSEC
         ret
        endif                   ;---------------------------------------
ioctl_8: cmp    bl,8            ;8 (dword volume size in sectors)
         jne    setbusy         ;unsupported IOCTL read 1..5, 9..15
         mov    si,XMSSEC
         xchg   si,cx           ;SI:CX = XMSSEC (2880 or 5760)

ioctlnn: mov    es:[di+1],cx
         mov    es:[di+3],si
         ret
ioctlrd  endp

;-----------------------------------------------------------------------
; device function 13h (generic IOCTL, device attribute bit 6)

generic  proc   near            ;AX 0, DX subunit, ES:DI request
         mov    cl,GENP+3       ;expect far ptr at ES:[DI+GENP]
         call   chkbuf
         call   genlook         ;generic IOCTL 840..84A, 860..86A func.
         jnz    badcmd          ;not zero if unknown function
         les    di,es:[di+GENP] ;ES:DI generic IOCTL parameters
         jmp    [si+1]          ;function offset follows code byte

badcmd   label  near
         mov    ax,8003h        ;status unknown function
         ret
generic  endp

chkbuf   proc   near            ;check non-minimal request length
         cmp    es:[di],cl
         ja     usebuf
         pop    ax              ;trash near return address
         mov    ax,8005h        ;bad request
usebuf:  ret
chkbuf   endp

chkout   proc   near            ;write access check (DX = 1, BP r/w)
         or     bp,bp
         jnz    allowio
         test   dskentry+11,dl  ;disk image read/only attribute
         jz     allowio
         pop    ax              ;trash near return address
         mov    ax,8000h        ;write-protect violation
allowio: ret
chkout   endp

;-----------------------------------------------------------------------
; device function 01h (media check)

check    proc   near            ;AX 0, DX subunit, ES:DI request
         mov    cx,MEDIA+1 + 256*(MEDIA+5)
         call   chkbuf          ;expect byte at ES:[DI+MEDIA+1]
         mov    byte ptr es:[di+MEDIA+1],1
         or     dx,dx
         jz     checked         ;subunit 0 never changed

         mov    si,offset label2
         xchg   dh,[si+11]      ;ASCIIZ NUL terminator
         or     dh,dh           ;reset and test label2
         jz     checked         ;no write access noted

         neg    byte ptr es:[di+MEDIA+1]
         cmp    es:[di],ch
         jna    checked         ;oops, cannot return label pointer
         mov    es:[di+MEDIA+2],si
         mov    es:[di+MEDIA+4],ds
checked: ret
check    endp

newlabel proc   near            ;refresh copy of volume label
         mov    si,offset volabel2+5
         mov    di,offset label2
         mov    cx,11           ;unclear:  does DOS want 11 bytes or
         rep    movsb           ;4+11+8 bytes serial + label + type ?
         mov    es:[di],dh      ;1: disk changed, 0: only new label
         ret                    ;(the terminating 0 is set by check)
newlabel endp

;-----------------------------------------------------------------------
; device function 02h (build BPB)

build    proc   near            ;AX 0, DX subunit, ES:DI request
         mov    cl,CONFIG+3     ;expect far ptr at ES:[DI+CONFIG]
         call   chkbuf
         inc    bp              ;BP 1: read access (1st FAT sector)
         mov    es:[di+CONFIG+2],ds
         or     dx,dx           ;build BPB, here return its address
         jnz    build2          ;subunit 1

         mov    word ptr es:[di+CONFIG],offset bpb
         call   lesdibuf        ;ES:DI = caller buffer
         inc    ax              ;start sector 1 (FAT)
         inc    dx              ;number of sectors: 1
         call   sysio           ;copy to buffer ES:DI 1st sector FAT
         xchg   ax,dx           ;remaining sectors: 0 (okay)
         ret

build2:  mov    word ptr es:[di+CONFIG],offset bpb2
         call   lesdibuf        ;ES:DI = caller buffer
         mov    opened2,ax      ;clear visible open counter
         mov    ax,dx           ;AX start sector 1 (FAT)
         jmp    tracxms         ;DX number of sectors: 1
build    endp

;-----------------------------------------------------------------------
; generic IOCTL 841h (write track) and 861h (read track), both subunits

trackio  proc   near            ;AX 0, DX subunit, BP query, ES:DI IOCTL
         call   chktrack        ;check head and track
         jc     bads_jc         ;(short jump badsect)

         mov    ax,es:[di+5]    ;start sector in track
         mov    si,es:[di+7]    ;number of sectors
         mov    cx,ax
         add    cx,si
bads_jc: jc     badsect
         or     dx,dx           ;subunit DX
         jz     tracmem

         call   chkout          ;write access check (DX = 1, BP r/w)
         mov    bx,XMSSEC/(2*80)
         cmp    cx,bx           ;18 (or EXTRA density 36)
         jnbe   badsect         ;(DOS buffer not checked)
         call   chs2abs         ;compute absolute sector number
         jmp    tracxms         ;same XMS code as for subunit 0

tracmem: mov    bx,nomulsec     ;44 per track (or 20 if EXTRA)
         cmp    cx,bx           ;MULSEC * LENSEC < 64 KB
         jnbe   badsect
         call   chs2abs         ;compute absolute sector number

tracker  label  near            ;code also used by read + write
         cmp    ax,SYSSEC       ;AX abs. sector, DX number
         jnb    sysskip
         call   sysio           ;move system data sectors

sysskip: sub    ax,SYSSEC
         mov    bx,LOWSEC+HMASEC
         sub    ax,bx           ;AX start relative to XMS
         jnb    tracxms         ;skip memory data sectors
         call   memio           ;move memory data sectors
         jnz    fatal           ;unexpected XMS A20 error
         mov    ax,cx           ;AX = CX = 0 if A20 okay

tracxms  label  near            ;code also used by subunit 1
         mov    cx,dx           ;number of sectors
         jcxz   moved           ;nothing to do
         shl    cx,PARSEC+4     ;sectors to bytes (0 for 64 KB)
         mov    dx,ax           ;start sector
         or     dx,bp           ;no write access if BP 1 (read)
         setz   bh              ;BH 1: write access on MBR
         mov    dx,LENSEC       ;DX bytes per sector, AX start
         mul    dx              ;linear offset DX:AX
         call   xmsio
         jcxz   fatal           ;XMS memory corrupted

         or     bh,bh           ;1: copy modified MBR to mbr2
         jz     moved           ;0: ready (MBR not modified)
         push   ds
         pop    es
         mov    di,offset mbr2  ;ES:DI target mbr2 contains bpb2
         mov    cx,LENFORM      ;length of mbr2 (= begin of MBR)
         inc    bp              ;BP 1 (read access)
         xor    ax,ax
         cwd                    ;DX:AX 0000:0000 from begin
         call   xmsio
         jcxz   fatal           ;XMS memory corrupted
         mov    dh,bh
         call   newlabel        ;store new label for check (DH = 1)

moved:   xor    ax,ax           ;okay
         ret
fatal:   mov    ax,800Ch        ;general failure
         ret
badsect: mov    ax,8008h        ;sector not found
         ret
trackio  endp

chs2abs  proc   near            ;CAVEAT: subunit used as head count
         push   si              ;expect input AX, BX, DX, SI, ES:DI
         mov    cx,dx           ;CX = heads - 1 (0 or 1)
         mov    si,bx           ;SI = SPT (sectors per track)
         mov    bx,es:[di+3]    ;BX = track (max. 545 on subunit 0)
         shl    bx,cl           ;BX = BX * 2 if two heads
         add    bx,es:[di+1]    ;BX = BX + 1 if 2nd head
         les    di,es:[di+9]    ;ES:DI = buffer
         xchg   ax,si           ;AX = SPT, SI = rel. start sector
         mul    bx              ;AX = SPT * tracks (max. 44 * 115)
         add    ax,si           ;AX = abs. start sector
         pop    dx              ;DX = number of sectors
         ret                    ;output AX, DX, ES:DI, (BP not touched)
chs2abs  endp

lesdibuf proc   near            ;eliminating two .386 jumps in ioctlrd
         les    di,es:[di+BUFFER]
         inc    di              ;\  / BUFFER offset -1 can be valid
         jz     fatal           ; ><  for EJECT, but is reported as
         dec    di              ;/  \ general driver failure 800Ch
         ret                    ;ES:DI = i/o buffer (flags modified)
lesdibuf endp

;-----------------------------------------------------------------------
; generic IOCTL 846h (write label) and 866h (read label), both subunits

volser   proc   near            ;AX 0, DX subunit, BP query, ES:DI IOCTL
         mov    si,offset volabel
         or     dx,dx           ;subunit DX
         jz     volsub
         call   chkout          ;write access check (DX = 1, BP r/w)
         mov    si,offset volabel2

volsub:  lodsb                  ;DS:SI boot sector volume id.
         cmp    al,28h
         jb     badvol
         cmp    al,29h          ;check 29h or obsolete 28h magic
         ja     badvol

         scasw                  ;ES:DI generic IOCTL offset 2
         mov    cx,23           ;4 vol. ser., 11 name, 8 type
         call   swapbuf         ;swap if BP 0 (write)
         rep    movsb
         call   swapbuf         ;swap if BP 0 (write)
         xchg   ax,cx           ;AX 0 (ready)
         or     dx,dx
         jz     volexit         ;ready if subunit 0

         push   ds              ;\/ ES = DS for oldlabel and
         pop    es              ;/\ mbr2 copy to XMS (ES:DI)
         call   newlabel        ;store new label for check (DH = 0)
         or     bp,bp
         jnz    volexit         ;ready if read access

         cwd                    ;DX:AX 0000:0000 (linear offset)
         mov    di,offset mbr2  ;ES:DI buffer, BP 0 (write XMS)
         mov    cx,LENFORM      ;length of mbr2
         call   xmsio
         jcxz   fatal           ;XMS memory corrupted
volexit: ret

badvol:  mov    ax,8007h        ;illegal medium: vol. id. not found,
         ret                    ;cf. gg244459.pdf (PC DOS 7 redbook)
volser   endp

;-----------------------------------------------------------------------
; generic IOCTL 840h (set param.s) and 860h (get param.s), subunit 1:

params   proc   near            ;AX 0, DX unit 1, BP query, ES:DI IOCTL
         xor    cx,cx
         mov    dh,es:[di]      ;get special function byte
         test   dh,11111010b    ;expect bit 76543.1. clear
         jnz    badparm         ;(*) no effect without these checks (?)
         mov    bx,offset devbpb2
         test   dh,1
         jz     usebpb2         ;use device BPB (bit 0 clear)
         mov    bx,offset bpb2  ;get medium BPB (bit 0 set)

usebpb2: xchg   ax,bx           ;AX bpb2, BX 0
         mov    si,offset params2
         call   swapbuf         ;swap if BP 0 (write)
         jnz    parmget         ;Z: write, NZ: read

         test   dh,00000100b    ;expect bit .....2.. set
         jz     badparm         ;(*) sector sizes cannot vary
         movsb                  ;00: copy bit 0, i.e., 0000010?b
         mov    cl,5            ;01: test immutable device type
         repe   cmpsb           ;02: test immutable attributes
         jne    badparm         ;04: test immutable max. track
         movsb                  ;06: copy density (0 default, 1 other)
         test   dh,1            ;bit 0 clear to define new device BPB
         jnz    samebpb         ;(*) no effect without these checks (?)
         xchg   ax,di           ;ES:DI devbpb2
         jmp    short parmbpb

parmget: mov    cl,7            ;(offset params2 + 7 is offset devbpb2)
         rep    movsb           ;copy device type + attributes + tracks:
         xchg   ax,si           ;DS:SI devbpb2 or medium bpb2

parmbpb: mov    cl,31           ;07: copy 31 bytes BPB (not checked)
         rep    movsb           ;38: physical track layout (ignored)
samebpb: xchg   ax,bx           ;AX 0 (okay)
         ret

badparm: mov    ax,800Fh        ;no change allowed
         ret
params   endp

;-----------------------------------------------------------------------
; generic IOCTL 842h (format track) and 862h (verify track), subunit 1:

forver   proc   near            ;AX 0, DX unit 1, BP query, ES:DI IOCTL
         mov    cl,2
         test   es:[di],cl      ;2 (bit 1): multitrack format or verify
         jnz    nomulti         ;multitrack not supported
         call   chktrack        ;check head and track
         jc     notrack
         call   chkout          ;write access check (DX = 1, BP r/w)
         dec    cx              ;0: okay
nomulti: dec    cx              ;1: not supported
notrack: mov    es:[di],cl      ;2: not allowed
         ret                    ;nothing happened, this is a RAM disk
forver   endp

chktrack proc   near            ;CAVEAT: subunit abused as head count
         mov    si,80
         or     dx,dx           ;subunit DX
         jnz    tracksi         ;subunit 1:  80 tracks (cylinders)
         mov    si,MEMSEC/MULSEC
         cmp    word ptr nomulsec,MULSEC
         je     tracksi         ;subunit 0: 115 tracks for 1.44 MB
         mov    si,nomemsec     ;subunit 0: 397 tracks for 2.88 MB
         shr    si,2            ;subunit 0: 513 tracks (for first MB)
tracksi: cmp    es:[di+1],dx    ;subunit 0: 545 tracks (HMA + 1st MB)
         jnbe   tracknc         ;+1 head number <= subunit
         cmp    es:[di+3],si    ;+3 track number < si (80, 115, etc.)
tracknc: cmc                    ;NC for B, C for NB (or NBE in jnbe)
         ret                    ;NC: okay, C: head or track incorrect
chktrack endp

;-----------------------------------------------------------------------
; generic IOCTL 847h (set access flag) and 867h (get flag), subunit 1:

drvflag  proc   near            ;AX 0, DX unit 1, BP query, ES:DI IOCTL
         mov    si,offset dskentry+11
         mov    dl,params2+3    ;DL device attributes (high byte)
         mov    dh,[si]         ;DH file attributes: bit 0 read/only
         mov    cx,2+100h       ;CL bit 1, CH bit 0
         or     bp,bp
         jnz    getflag

         or     dx,cx           ;access and read only bits set
         cmp    es:[di+1],al    ;0: disallow access (not formatted)
         je     setflag
         xor    dx,cx           ;access and read only bits cleared
setflag: mov    params2+3,dl    ;modify bit 9 in device attributes
         mov    [si],dh         ;modify bit 0 in image file attributes

getflag: test   dl,cl           ;TBD: is this access flag relevant for
         setz   es:[di+1]       ;removable media ?  So far it is only
         ret                    ;noted, but not actually used anywhere
drvflag  endp

;-----------------------------------------------------------------------
; generic IOCTL 848h (locking) and 868h (sense media), both subunits

senslock proc   near            ;AX 0, DX subunit, BP query, ES:DI IOCTL
         or     bp,bp           ;1: get (868h)
         jz     locking         ;0: n/a (848h), locking operations
         or     dx,dx           ;subunit DX
         mov    dh,5            ;sububit 0 is a fixed disk (5)
         jz     fixedid         ;CAVEAT: subunit DL used as media flag
         mov    dh,DISKTYPE     ;subunit 1 is 1440 (7) or 2880 (9)
fixedid: mov    es:[di],dx      ;keep DL 1 (default media) or 0 (other)
         ret

jmpbusy: jmp    setbusy         ;use "busy" to report ignored issues
locking: mov    bx,offset sublocks
         add    bx,dx           ;+0 subunit 0, +1 subunit 1
         mov    cl,[bx]
         mov    dl,es:[di+0]
         cmp    dl,2
         je     lockget         ;2: query pending locks
         jnb    jmpbusy         ;expected 0, 1, or 2
         add    dx,dx           ;1: decrement locks \/ 1*2 -1 = +1
         dec    dx              ;0: increment locks /\ 0*2 -1 = -1
         sub    cl,dl
         jc     jmpbusy         ;locking overflow or underflow
         mov    [bx],cl
lockget: mov    es:[di+1],cl
         ret
senslock endp

;-----------------------------------------------------------------------
; generic IOCTL 849h (eject) and 869h (busy), subunit 1, no parameters
; TBD: should eject fail if the logical volume is locked ?

eject    proc   near            ;AX 0, DX unit 1, BP query, ES:DI n/a
         or     bp,bp           ;there is no 869h, but MEMDRIVE handles
         jnz    jmpbusy         ;all generic IOCTLs as set + get pairs

         mov    si,offset dskentry+11
         test   byte ptr [si],LOCKED
         jnz    jmpbusy         ;door locked, cannot eject disk image

         mov    [si],al         ;0: clear disk image attributes
         mov    sublocks+1,al   ;0: clear pending drive locks
         mov    opened2,ax      ;0: clear visible open counter
         mov    label2+11,dl    ;1: note disk change for check
         ret
eject    endp

;-----------------------------------------------------------------------
; generic IOCTL 84Ah (lock) and 86Ah (unlock), subunit 1, no parameters

fdlock   proc   near            ;AX 0, DX unit 1, BP query, ES:DI n/a
         mov    bx,offset sublocks+1
         or     bp,bp           ;BP Z/NZ often means set or get (query),
         mov    dl,255          ;here it means lock or unlock subunit 1
         jz     fdlock1         ;expect DH 0, want DL 255 (= lock)
         xchg   dh,dl           ;expect DH 255, want DL 0 (= unlock)
fdlock1: cmp    [bx],dh
         jne    jmpbusy         ;an erroneous IOCTL 848h decrement from
         mov    [bx],dl         ;255 to 254 would cause spurious errors
         ret
fdlock   endp

;-----------------------------------------------------------------------
; R/W access on allocated XMS memory: DX:AX linear offset, ES:DI buffer,
; CX length (bytes), BP r/w.  Result CX 0 error (code in BL), else okay.

xmsio    proc   near
         push   bx
         mov    si,offset xmspara
         mov    [si+0],cx       ;block length low
         xor    bx,bx
         cmp    bx,cx           ;CX 0: length 64 KB
         mov    cx,bx
         sete   bl
         mov    [si+2],bx       ;block length high

         mov    bl,8            ;BX 08 source, 14 target
         or     bp,bp           ;1: read  (XMS to ES:DI)
         jz     xmsin           ;0: write (ES:DI to XMS)
         xor    bl,6
xmsin:   mov    [bx+si-4],cx    ;CX 0: low memory buffer
         mov    [bx+si-2],di    ;ES:DI low target/source
         mov    [bx+si-0],es
         xor    bl,6            ;BX 14 target, 08 source
         mov    cx,xmsdata
         mov    [bx+si-4],cx    ;CX is handle XMS memory
         mov    [bx+si-2],ax    ;DX:AX linear XMS offset
         mov    [bx+si-0],dx
         pop    bx

         xchg   ax,cx           ;store AX
         mov    ah,11           ;copy XMS block
         call   [xms]           ;DS:SI description
         xchg   ax,cx           ;reset AX
         jcxz   xmserr
         ret                    ;CX 1: okay (BX preserved)
xmserr:  mov    xmsdiag,bl      ;CX 0: error (BL modified)
         ret                    ;SI modified
xmsio    endp

;-----------------------------------------------------------------------
; R/W access on first MB or HMA, input parameters ES:DI buffer, AX start
; sector (fixed here), DX sector count, BP r/w.  Output ES:DI, AX, and
; DX adjusted after copying memory sectors, BP preserved.

; Example:  Start sector 2144 is linear 10C000h.  Paragraph 1.0C00h is
; converted to FFFF:C010.  Sector count 32 means 4000h bytes ending at
; 110000h.  The last paragraph before 1.1000h is 1.0FFFh and cannot be
; addressed in segment FFFF.  The last 16 bytes are ignored for write or
; emulated for read access (to check this use a disk editor).  In fact
; the HMA consists of 64 KB - 16 = 65520 bytes; cf. HMA.DAT file size.

memio    proc   near
         add    ax,bx           ;undo AX - BX by caller
         mov    cx,ax           ;start sector in memory
         add    cx,dx           ;start + count
         sub    cx,bx           ;start + count - memory
         jnc    memrest
         xor    cx,cx           ;no rest if CX < 0
memrest: sub    dx,cx           ;count low + HMA sectors
         push   cx              ;count remaining sectors

         shl    dx,PARSEC+2     ;2**(5+2)=128 per sector
         mov    cx,dx           ;CX = DX dwords
         jcxz   memexit         ;nothing to do

         xor    si,si           ;SI for a source DS:0
         mov    dx,LENSEC/16    ;paragraphs per sector
         mul    dx              ;start paragraph in AX
         jnc    memcopy         ;start still below HMA

         mov    dx,cx           ;CX = DX dwords
         xchg   ax,si           ;determine HMA address
         dec    ax              ;AX FFFF start segment
         inc    si              ;SI 0001 up to 0FFF
         shl    si,4            ;SI 0010 up to FFF0
         mov    cx,dx           ;CX = DX dwords
         shl    dx,2            ;number of bytes
         add    dx,si           ;last accessed offset
         jnc    memcopy
         sub    cx,4            ;protect last 16 bytes (4 dwords)
         shl    bp,1            ;BP 0: write, 1: read, 2: read HMA

memcopy: mov    ds,ax           ;source address DS:SI
         assume ds:nothing
         call   swapbuf         ;swap if BP 0 (write)

         mov    ah,5            ;A20 on
         call   cs:[xms]
         dec    ax
         jnz    a20fail         ;-1: XMS error
         rep    movsd
         mov    ah,6            ;A20 off
         call   cs:[xms]

a20fail: cmp    bp,2            ;check read access flag
         jne    memdone         ;last paragraph is okay
         mov    cx,8            ;emulate last 16 bytes
         mov    si,offset paranoia
         rep    movs word ptr [di],word ptr cs:[si]
         dec    bp              ;restore read flag

memdone: call   swapbuf         ;swap if BP 0 (write)
         push   cs
         pop    ds              ;restore own segment
         assume ds:DGROUP
         dec    ax
         jz     memexit         ;-1 or -2 A20 error
         mov    xmsdiag,bl      ;note XMS error code

memexit: pop    dx              ;adjusted rest count
         or     ch,cl           ;Z: okay (CX 0)
         ret                    ;NZ: A20 error
memio    endp

;-----------------------------------------------------------------------
sysmove  proc   near            ;sysmove belongs to sysio in the OVERLAY
         call   swapbuf         ;swap if BP 0 (write)
         rep    movsw           ;number of words in CX
         call   swapbuf         ;swap if BP 0 (write)
         ret
sysmove  endp

swapbuf  proc   near
         or     bp,bp
         jnz    reading         ;NZ: read access
         push   ds              ;Z: write access
         push   es
         pop    ds
         pop    es              ;swap es, ds
         xchg   si,di           ;swap si, di
reading: ret                    ;Z: write, NZ: read
swapbuf  endp

;-----------------------------------------------------------------------
         align  16              ;odd alignment would corrupt init stack

sector0  label  near
         jmp    short reboot1   ;boot sector              ; 00   0
         nop                    ;                         ; 02   2
         db     "MEMDRIVE"      ;manufacturer             ; 03   3
bpb      dw     LENSEC          ;sector size in bytes     ; 0B  11 (+02)
         db     64              ;64 sectors per cluster   ; 0D  13 (+03)
         dw     1               ;1 reserved = this sector ; 0E  14 (+05)
         db     2               ;2 FATs (mostly virtual)  ; 10  16 (+06)
         dw     16              ;root directory entries   ; 11  17 (+08)
nomemsec dw     MEMSEC          ;sector number  <= 32 MB  ; 13  19 (+10)
         db     MEDIA1          ;media descriptor         ; 15  21 (+11)
         dw     1               ;FAT sectors              ; 16  22 (+13)
nomulsec dw     MULSEC          ;sectors per track        ; 18  24 (+15)
         dw     1               ;number of heads          ; 1A  26 (+17)
         dd     0               ;hidden sectors           ; 1C  28 (+21)
         dd     0               ;sector number   > 32 MB  ; 20  32 (=25)
         dw     0               ;physical drive, 0 or 128 ; 24  36
volabel  db     41              ;41 DOS 4 & above format: ; 26  38
         dd     12345678h       ; 4 bytes volume serial   ; 27  39
         db     "MEM drive ?"   ;11 bytes volume name     ; 2B  43
         db     "FAT12   "      ; 8 bytes volume type     ; 36  54
reboot1: int    18h             ;dummy boot routine       ; 3E  62
OVERLAY  equ    $
         db     228 dup (?)     ;fill 228 = 512 - 64 - 220; 40  64
message  db     "MEM drive ? installed: (c) 2010"         ;124 292 220
         db     " by F. Ellermann"                        ;143 323 189
         db     13,10,"$"                                 ;153 339 173
xmsdiag  db     0               ;last XMS error code      ;156 342 170
label2   db     12 dup (0)      ;ASCIIZ volabel2+5 copy   ;157 343 169
params2  db     FIXTRACK        ;860h may toggle bit 0    ;163 353 157
         db     DISKTYPE        ;7 allows FORMAT /F:1.44  ;164 356 156
         dw     DOORLOCK        ;door lock support        ;165 357 155
         dw     80              ;immutable track number   ;167 359 153
         db     0               ;default density          ;168 361 151
devbpb2  dw     LENSEC          ;sector size in bytes     ;17A 362 (+02)
         db     CLUSTER         ;sectors per cluster      ;16C 364 (+03)
         dw     1               ;1 reserved = this sector ;16D 365 (+05)
         db     2               ;2 FATs                   ;16F 367 (+06)
         dw     DIRSEC*LENSEC/32        ;root dir entries ;170 368 (+08)
         dw     XMSSEC          ;sector number  <= 32 MB  ;172 370 (+10)
         db     MEDIA2          ;media descriptor         ;173 372 (+11)
         dw     FATSEC          ;FAT sectors              ;175 373 (+13)
         dw     XMSSEC/(2*80)   ;sectors per track        ;177 375 (+15)
         dw     2               ;two heads for 3.5 floppy ;179 376 (+17)
         dd     0               ;hidden sectors           ;17B 379 (+21)
         dd     0               ;sector number   > 32 MB  ;17F 383 (+25)
         db     FIXTRACK        ;six extended BPB bytes:  ;183 387 (+26)
opened2  dw     0               ; open counter  (unclear) ;184 388 (+28)
         db     DISKTYPE        ; same as above (unclear) ;186 390 (+29)
         dw     DOORLOCK        ; same as above (unclear) ;187 391 (=31)
genlis2  db     40h             ;generic IOCTL 840h/860h  ;188 393 119
         dw     offset params   ; set or get parameters   ;18A 394 118
         db     42h             ;generic IOCTL 842h/862h  ;18B 396 116
         dw     offset forver   ; format or verify track  ;18D 397 115
         db     47h             ;generic IOCTL 847h/867h  ;18E 399 113
         dw     offset drvflag  ; set or get access flag  ;190 400 112
         db     49h             ;generic IOCTL 849h       ;192 402 110
         dw     offset eject    ; eject removable media   ;193 403 109
         db     4Ah             ;generic IOCTL 84Ah/86Ah  ;195 405 108
         dw     offset fdlock   ; FreeDOS format.exe lock ;196 406 106
genlist  db     41h             ;generic IOCTL 841h/861h  ;198 408 104
         dw     offset trackio  ; write or read track     ;199 409 103
         db     46h             ;generic IOCTL 846h/866h  ;19B 411 101
         dw     offset volser   ; set or get volume label ;19C 412 100
         db     48h             ;generic IOCTL 848h/868h  ;19D 414  98
         dw     offset senslock ; media 5/7/9, lock 0/1/2 ;19F 415  97
MAXIOCTL equ    (($ - offset genlist) / 3)      ;excludes genlis2
MAXIOCT2 equ    (($ - offset genlis2) / 3)      ;includes genlist
running  db     -1              ;global access semaphore  ;1A1 417  95
sublocks db     0,0             ;lock counter subunit 0+1 ;1A2 418  94
xmspara  dd     0               ;XMS block length (bytes) ;1A4 420  92
         dw     0               ;source handle or 0       ;1A8 424  88
         dd     0               ;source offset or address ;1AA 426  86
         dw     0               ;target handle or 0       ;1AE 430  82
         dd     0               ;target offset or address ;1B0 432  80
xmsdata  dw     -1              ;XMS handle data sectors  ;1B4 436  76
commands dw     offset init     ; 0: initialization       ;1B6 438  74
         dw     offset check    ; 1: medium test          ;1B8 440  72
         dw     offset build    ; 2: build BPB            ;1BA 442  70
         dw     offset ioctlrd  ; 3:  input IOCTL         ;1BC 444  68
         dw     offset input    ; 4:  input               ;1BE 446  66
         dw     offset badcmd   ; 5:  input no wait       ;1C0 448  64
         dw     offset badcmd   ; 6:  input status        ;1C2 450  62
         dw     offset badcmd   ; 7:  input flush         ;1C4 452  60
         dw     offset write    ; 8: output               ;1C6 454  58
         dw     offset write    ; 9: output verify        ;1C8 456  56
         dw     offset badcmd   ;10: output status        ;1CA 458  54
         dw     offset badcmd   ;11: output flush         ;1CC 460  52
         dw     offset ioctlwr  ;12: output IOCTL         ;1CE 462  50
         dw     offset open     ;13: Open                 ;1D0 464  48
         dw     offset close    ;14: Close                ;1D2 466  46
         dw     offset fixed    ;15: Removeable media     ;1D4 468  44
         dw     offset badcmd   ;16: output until busy    ;1D6 470  42
         dw     offset badcmd   ;17: Siemens DOS 4.0 con  ;1D8 472  40
         dw     offset badcmd   ;18: Siemens DOS 4.0 con  ;1DA 474  38
         dw     offset generic  ;19: generic IOCTL func.  ;1DC 476  36
         dw     offset badcmd   ;20: KKCFUNC DOS 4.0 con  ;1DE 478  34
         dw     offset badcmd   ;21: Siemens DOS 4.0      ;1E0 480  32
         dw     offset badcmd   ;22: unknown DOS 4.0      ;1E2 482  30
         dw     offset getset   ;23: get logical device   ;1E4 484  28
         dw     offset getset   ;24: set logical device   ;1E6 486  26
         dw     offset gentest  ;25: generic IOCTL check  ;1E8 488  24
UNKNOWN  equ    ($ - offset commands) shr 1
request  dd     -1              ;request block address    ;1EA 490  22
oldstack dd     -1              ;stack of caller          ;1EE 494  18
newstack dd     offset stacktop ;segments fixed by init:  ;1F2 498  14
xms      dd     offset xmsdummy ;XMS address when loaded  ;1F6 502  10
bpbptr   dw     offset bpb      ;1st BPB                  ;1FA 506   6
         dw     offset bpb2     ;2nd BPB                  ;1FC 508   4
         dw     MAGIC           ;boot sector end mark     ;1FE 510   2

sector1  db     MEDIA1,255,255  ;two reserved clusters FAT12
         db     003h, 40h, 00h  ;  3 0000   6 clusters 2+3
         db     005h, 60h, 00h  ;  5 1000   9 clusters 4+5
         db     007h, 80h, 00h  ;  7 2000  12 etc.
         db     009h,0A0h, 00h  ;  9 3000  15 memory 3000:0000
         db     00Bh,0C0h, 00h  ; 11 4000  18 memory 4000:0000
         db     00Dh,0E0h, 00h  ; 13 5000  21 etc.
         db     00Fh, 00h, 01h  ; 15 6000  24 bytes 21, 22, 23
         db     011h, 20h, 01h  ; 17 7000  27 bytes 24, 25, 26
         db     013h, 40h, 01h  ; 19 8000  30 etc.
         db     015h, 60h, 01h  ; 21 9000  33
         db     017h, 80h, 01h  ; 23 A000  36
         db     019h,0A0h, 01h  ; 25 B000  39
         db     01Bh,0C0h, 01h  ; 27 C000  42
         db     01Dh,0E0h, 01h  ; 29 D000  45
         db     01Fh, 00h, 02h  ; 31 E000  48
         db     021h,240 ,255   ; 33 F000  51
         db     023h,240 ,255   ; 35 0000  54
        ;       021h, 20h, 02h  ;EOF 1st MB
        ;       023h, 40h, 02h  ;EOF HMA
USEFAT   equ    ($ - offset sector1)
.errnz   (USEFAT - (USEFAT/2)*2)        ;even required in sysfat
         db     025h, 60h, 02h  ; 37   64  57 clusters 36+37
         db     027h, 80h, 02h  ; 39  128  60 clusters 38+39
         db     029h,0A0h, 02h  ; 41  192  63 etc.
         db     02Bh,0C0h, 02h  ; 43  256  66 floppy image 256 KB
         db     02Dh,0E0h, 02h  ; 45  320  69 floppy image 320 KB
         db     02Fh, 00h, 03h  ; 47  384  72 etc.
         db     031h, 20h, 03h  ; 49  448  75 bytes 72, 73, 74
         db     033h, 40h, 03h  ; 51  512  78 bytes 75, 76, 77
         db     035h, 60h, 03h  ; 53  576  81 etc.
         db     037h, 80h, 03h  ; 55  640  84
         db     039h,0A0h, 03h  ; 57  704  87
         db     03Bh,0C0h, 03h  ; 59  768  90
         db     03Dh,0E0h, 03h  ; 61  832  93
         db     03Fh, 00h, 04h  ; 63  896  96
         db     041h, 20h, 04h  ; 65  960  99
         db     043h, 40h, 04h  ; 67 1024 102
         db     045h, 60h, 04h  ; 69 1088 105
         db     047h, 80h, 04h  ; 71 1152 108
         db     049h,0A0h, 04h  ; 73 1216 111
         db     04Bh,0C0h, 04h  ; 75 1280 114
         db     04Dh,0E0h, 04h  ; 77 1344 117
         db     04Fh, 00h, 05h  ; 79 1408 120
        ifndef  EXTRA           ; high density (HD) image end
         db      255,15  ,0     ;EOF 1472 123 (end is at 1440)
         db     5 dup (0)       ;fill 5 + 123 = 128 (align 16)
        else                    ;extra density (ED) image:
         db     051h, 20h, 05h  ; 81 1472 123
         db     053h, 40h, 05h  ; 83 1536 126
         db     055h, 60h, 05h  ; 85 1600 129
         db     057h, 80h, 05h  ; 87 1664 132
         db     059h,0A0h, 05h  ; 89 1728 135
         db     05Bh,0C0h, 05h  ; 91 1792 138
         db     05Dh,0E0h, 05h  ; 93 1856 141
         db     05Fh, 00h, 06h  ; 95 1920 144
         db     061h, 20h, 06h  ; 97 1984 147
         db     063h, 40h, 06h  ; 99 2048 150
         db     065h, 60h, 06h  ;101 2112 153
         db     067h, 80h, 06h  ;103 2176 156
         db     069h,0A0h, 06h  ;105 2240 159
         db     06Bh,0C0h, 06h  ;107 2304 162
         db     06Dh,0E0h, 06h  ;109 2368 165
         db     06Fh, 00h, 07h  ;111 2432 168
         db     071h, 20h, 07h  ;113 2496 171
         db     073h, 40h, 07h  ;115 2560 174
         db     075h, 60h, 07h  ;117 2624 177
         db     077h, 80h, 07h  ;119 2688 180
         db     079h,0A0h, 07h  ;121 2752 183
         db     07Bh,0C0h, 07h  ;123 2816 186
        ;       07Dh,0E0h, 07h  ;EOF 2880 KB
         db     07Dh,240 ,255   ;125 2880 189
         db     3 dup (0)       ;fill 3 + 189 = 192 (align 16)
        endif
LENFAT   equ    ($ - offset sector1)

sector3  label  near
volume1  db     "MEM drive ?",8 ;======== 1st entry ========
         db     10 dup (0)      ;attribute 8 (volume label)
         dw     (12 shl 11)+(34 shl 5)+(56 shr 1)       ;time
         dw     ((2010-1980) shl 9)+(07 shl 5)+07       ;date
         dw     0               ;no clusters for volume label
         dd     0               ;length 0
dskentry db     DISKNAME,"IMG",0;======== 2nd entry ========
         db     10 dup (0)      ;attribute 0 (write access okay)
time1    dw     -1              ;[DOS root directory write access
date1    dw     -1              ; up to USEDIR, rest is read-only]
         dw     36              ;45 clusters 36..80 after HMA.DAT
         dd     LENSEC * XMSSEC ;1440 KB (if XMS available)
USEDIR   equ    ($ - offset sector3)
         db     "MEMORY  DAT",5 ;======== 3rd entry ========
         db     10 dup (0)      ;attribute 5 (system + read only)
         dw     6000h           ;[DOS root directory read access
date2    dw     -1              ; up to LENDIR, rest is emulated]
         dw     2               ;32 clusters 2..33 after FAT12 id.
         dd     LENSEC * LOWSEC ;1 MB (incl. ROM)
hmaentry db     "HMA     DAT",5 ;======== 4th entry ========
         db     10 dup (0)      ;attribute 5 (system + read only)
time3    dw     -1              ;[12 = 16 - 4 = LENSEC / 32 - 4
date3    dw     -1              ; remaining entries are emulated]
         dw     34              ;clusters 34+35 after MEMORY.DAT
         dd     LENSEC * HMASEC - 16
LENDIR   equ    ($ - offset sector3)
paranoia db     "(OUTSIDE OF HMA)"

mbr2     label  near            ;2nd subunit in XMS (copied by format)
         jmp    short reboot2   ;DOS checks the jump here ; 00   0
         nop                    ;                         ; 02   2
         db     "MEMDRIVE"      ;manufacturer             ; 03   3
bpb2     dw     LENSEC          ;sector size in bytes     ; 0B  11 BPB
         db     CLUSTER         ;sectors per cluster      ; 0D  13 BPB
         dw     1               ;1 reserved = this sector ; 0E  14 BPB
         db     2               ;2 FATs                   ; 10  16 BPB
         dw     DIRSEC*LENSEC/32        ;root dir entries ; 11  17 BPB
         dw     XMSSEC          ;sector number  <= 32 MB  ; 13  19 BPB
         db     MEDIA2          ;media descriptor         ; 15  21 BPB
         dw     FATSEC          ;FAT sectors              ; 16  22 BPB
         dw     XMSSEC/(2*80)   ;sectors per track        ; 18  24 BPB
         dw     2               ;two heads for 3.5 floppy ; 1A  26 BPB
         dd     0               ;hidden sectors           ; 1C  28 BPB
         dd     0               ;sector number   > 32 MB  ; 20  32 BPB
         dw     0               ;physical drive, 0 or 128 ; 24  36
volabel2 db     41              ;41 DOS 4 & above format: ; 26  38
         dd     12345678h       ; 4 bytes volume serial   ; 27  39
         db     "MEM drive ?"   ;11 bytes volume name     ; 2B  43
         db     FATNAME         ; 8 bytes volume type     ; 36  54
reboot2: int    18h             ;format aligned paragraph ; 3E  62
LENFORM  equ    ($ - offset mbr2)                         ; 40  64

bottom   db     " MEMDRIVE stack "
         org    $+12*16
stacktop label  byte            ;stack size 208 = 16 * (12+1)
         org    $-12*16

;-----------------------------------------------------------------------
; init uses the stack of the caller.  format overwrites LENFORM bytes of
; init, later init overwrites itself up to stacktop (two .errnz checks)

         assume ds:DGROUP, ss:nothing

init     proc   near            ;device initialization (later the stack)
         mov    word ptr xms+2,cs
         mov    word ptr newstack+2,cs
         mov    cl,INIERR       ;expect byte ES:[DI+INIERR]
         call   chkbuf          ;(implicitly asserts DOS 5+)

         mov    word ptr commands,offset badcmd
         push   ds
         push   es
         pop    ds              ;DS = request segment
         assume ds:nothing
         mov    ax,0241h        ;AH: units, AL: drive letter
         add    al,[di+VOLID]   ;drive number (above DOS 3.0)
         mov    [di+MEDIA],ah
         mov    [di+CONFIG+0],offset bpbptr
         mov    [di+CONFIG+2],cs
         mov    [di+BUFFER+0],offset stacktop
         mov    [di+BUFFER+2],cs
         pop    ds              ;DS = own segment
         assume ds:DGROUP

         mov    byte ptr volume1+10,al
         mov    byte ptr message+10,al
         mov    si,offset volabel+1
         mov    [si+14],al      ;= volabel+15
         push   si
         inc    ax              ;for second volume
         mov    si,offset volabel2+1
         mov    [si+14],al      ;= volabel2+15

         call   datetime        ;CX ccyy, DX oodd, BX hhmm, AL ss
         mov    [si+0],dx
         mov    [si+2],cx       ;VOL displays this as ccyy-oodd
         pop    si
         mov    [si+0],al       ;pseudo-random serial number
         mov    [si+1],bx
         mov    [si+3],dl       ;VOL displays this as ddhh-mmss

         call   filetime        ;BX:AL to AX, enforces even seconds
         mov    time1,ax        ;CX:DX to DX, 1980 <= yyyy <= 2107
         mov    time3,ax
         xchg   ax,dx
         mov    date1,ax
         mov    date3,ax
         call   biosdate        ;CX 20yy, DX mmdd (BIOS F000:FFF5)
         call   filedate        ;CX:DX to AX, 1980 <= 20yy <= 2107
         mov    date2,ax        ;hardwired BIOS time 12:00 (6000h)

         mov    ax,4300h        ;test XMS presence
         int    2Fh
         cmp    al,80h
         jne    initlow         ;no XMS, also no HMA

         push   es
         mov    ax,4310h        ;get XMS entry point
         int    2Fh
         mov    word ptr xms+0,bx
         mov    word ptr xms+2,es
         pop    es

         mov    ah,9            ;allocate XMS memory
         mov    dx,XMSSEC * LENSEC / 1024
         call   dword ptr [xms]
         or     ah,al
         jz     inithma         ;no XMS, but use HMA

         mov    xmsdata,dx      ;note XMS data handle
         push   es
         push   di
         call   format
.errnz   ($ - offset init) lt LENFORM
         pop    di
         pop    es              ;ES:DI request
         jcxz   initxms         ;CX = 0 (okay)

         mov    dx,offset msgerr
         mov    ah,9
         int    21h
         mov    word ptr es:[di+BUFFER],0
         inc    byte ptr es:[di+INIERR]
         mov    ax,800Ch        ;general failure
         ret

msgerr   db     13,10,DISKNAME,": XMS error",07,13,10,"$"

initlow: xor    ax,ax
         mov    word ptr xms+2,cs       ;CS far ptr to xmsdummy
         sub    nomemsec,HMASEC         ;delete HMA sectors
         mov    bx,offset hmaentry
         mov    byte ptr [bx],DELETED   ;delete HMA file
         mov    word ptr [bx+26],ax     ;no old HMA start cluster
         mov    bx,ax                   ;BL = 0 no XMS error

inithma: mov    xmsdiag,bl              ;note XMS error code
         dec    byte ptr es:[di+MEDIA]  ;delete 2nd unit
         sub    nomemsec,XMSSEC         ;delete XMS sectors
         mov    nomulsec,4              ;adjust sectors per track
         mov    bx,offset dskentry
         mov    byte ptr [bx],DELETED   ;delete DSK (disk image)
         mov    word ptr [bx+26],ax     ;no old DSK start cluster

initxms: mov    dx,offset message
         mov    ah,9
         int    21h

         push   ds
         pop    es              ;ES:DI stack bottom
         mov    di,offset bottom
         mov    cx,(offset stacktop - offset bottom)/16

stacked: xchg   ax,cx           ;code overwritten by REP MOVSW
         mov    si,offset bottom
         mov    cx,8            ;8 words from DS:SI bottom
         rep    movsw           ;to ES:DI bottom (ES = DS)
.errnz   (offset stacktop - $) gt 0
         xchg   ax,cx
         loop   stacked         ;overwrite CX paragraphs
         ret                    ;AX = 0
init     endp

;-----------------------------------------------------------------------
; BIOS date + time (BCD timestamp), result CX:DX:BX:AL ccyy:oodd:hhmm:ss

datetime proc   near
         mov    ah,4            ;4: date (rtcread toggle)
         call   near rtcread    ;2: time CX hhmm, DH ss
         mov    bx,cx
         mov    al,dh           ;BX:AL = CX:DH (hhmm:ss)
         call   near rtcread    ;4: date CX ccyy, DX oodd
         push   cx
         push   dx              ;store CX:DX (ccyy:oodd)
         call   near rtcread    ;2: time CX hhmm, DH ss
         cmp    dh,al
         pop    dx
         pop    cx              ;reset CX:DX (ccyy:oodd)
         jne    datetime        ;try again if new second
         ret                    ;CX:DX:BX:AL ccyy:oodd:hhmm:ss

rtcread: xor    ah,6            ;4:2 (time), 2:4 (date)
rtcwait: clc                    ;NC avoids INT 1Ah bug
         push   ax
         int    1Ah
         pop    ax
         jc     rtcwait         ;C: update in progress
         retn                   ;near inline proc
datetime endp

filetime proc   near
         push   ax              ;time BX hhmm, AL ss
         call   filedate        ;date CX ccyy, DX oodd
         xchg   ax,dx           ;put file date in DX
         xchg   ax,bx           ;time AX hh:mm
         mov    ch,ah           ;AL = mm (BCD)
         call   bcd2int         ;AX = mm (int)
         shl    ax,5
         push   ax              ;DOS .....mmmmmm54321

         mov    al,ch           ;AL = hh (BCD)
         call   bcd2int         ;AX = hh (int)
         shl    ax,11           ;DOS hhhhh10987654321

         pop    cx              ;DOS .....mmmmmm.....
         add    cx,ax           ;DOS hhhhhmmmmmm.....
         pop    ax              ;AL = ss (BCD)
         call   bcd2int         ;AX = ss (int)
         shr    ax,1            ;seconds/2 for 5 bits
         add    ax,cx           ;DOS hhhhhmmmmmmsssss
         ret                    ;DOS timestamp DX:AX
filetime endp

filedate proc   near            ;date CX ccyy, DX oodd
         mov    al,cl           ;AL = yy (BCD)
         call   bcd2int         ;AX = yy (int)
         xchg   ax,cx           ;CX = yy, AX = ccyy
         mov    al,ah           ;AL = cc (BCD)
         call   bcd2int         ;AX = cc (int)
         imul   ax,100          ;AX = cc * 100
         add    ax,cx           ;AX = ccyy (int)
         sub    ax,1980         ;AX = years since 1980
         shl    ax,9            ;DOS limit 127 is 2107
         push   ax              ;DOS yyyyyyy987654321

         mov    al,dh           ;AL = oo (BCD month)
         call   bcd2int         ;AX = oo (int month)
         shl    ax,5            ;DOS .......oooo54321

         pop    cx
         add    cx,ax           ;DOS yyyyyyyoooo.....
         mov    al,dl           ;AL = dd (BCD)
         call   bcd2int         ;AX = dd (int)
         add    ax,cx           ;DOS yyyyyyyooooddddd
         ret                    ;AX file date
filedate endp

bcd2int  proc   near
        ifndef  PEDANTIC
         aam                    ;AH = AL / 10 ; AL = AL mod 10
         org    $-1             ;AAM patched ("undocumented"):
         db     16              ;AH = AL / 16 ; AL = AL mod 16
        else
         mov    ah,16
         mul    ah              ;AX 10xy to 0xy0
         shr    al,4            ;AX 0xy0 to 0x0y
        endif
         aad                    ;AL = AH * 10 + AL ; AH = 0
         ret
bcd2int  endp

biosdate proc   near
         push   ds
         mov    cx,0F004h       ;CL 4 used for AX rotations
         mov    ds,cx           ;[Phoenix US ASCII timestamp]
         mov    si,0FFB5h       ;F000:FFF5 = F004:FFB5
         call   near bios2ah    ;F000:FFF5 (3m3M in Mm/Dd/Yy)
         mov    dh,ah           ;DH = Mm
         call   near bios2ah    ;F000:FFF8 (3d3D in Mm/Dd/Yy)
         mov    dl,ah           ;DL = Dd
         call   near bios2ah    ;F000:FFFB (3y3Y in Mm/Dd/Yy)
         mov    cl,ah           ;CL = Yy
         mov    ch,20h          ;assume BIOS year 2000..2099
         pop    ds              ;CX 20Yy, DX MmDd (BCD)
         ret

bios2ah: lodsw                  ;AX 3R3L (10*L+R, ASCII 3x)
         ror    ax,cl           ;AX L3R3
         ror    ah,cl           ;AX 3LR3
         rol    ax,cl           ;AX LR33
         lodsb                  ;AX LR2F (BCD result in AH)
         retn                   ;near inline proc
biosdate endp

;-----------------------------------------------------------------------
; CAVEAT, format abuses the first LENFORM init code bytes as a buffer:
; Quick format of the initially empty RAM disk (assuming F0h, 1440 KB)

format   proc   near
         push   ds
         pop    es
         mov    di,offset init  ;clobber ES:DI init code
         mov    bx,LENFORM      ;size of format buffer
         mov    al,246          ;F6h fill byte for format
         mov    cx,bx
         rep    stosb           ;LENFORM F6h (246) bytes
         xor    bp,bp           ;BP 0 for xmsio write access
         xchg   ax,cx           ;AX 0
         cwd                    ;DX:AX 0000:0000 (i.e. from begin)
         mov    cx,XMSSEC*LENSEC/LENFORM
         mov    di,offset init
         call   near formit     ;overwrite all (CX buffer copies)

         xchg   ax,cx           ;AX 0
         cwd                    ;DX:AX 0000:0000 (i.e. from begin)
         mov    cx,bx
         rep    stosb           ;LENFORM NUL bytes
         mov    cx,(1+2*FATSEC+DIRSEC)*LENSEC/LENFORM
         mov    di,offset init
         call   near formit     ;zero MBR + FATs + root dir.

         mov    si,offset volume1
         mov    cx,32           ;copy 1st directory entry (label)
         rep    movsb           ;DX:AX 0000:2600 (1 + 2*9 FATSEC)
         mov    ax,LENSEC*(1+2*FATSEC)
         mov    di,offset init
         inc    byte ptr [di+10]
         inc    cx              ;CX 1 copy of LENFORM bytes at ES:DI
         call   near formit     ;DOS 7 wants a volume label entry (?)

         xchg   ax,cx
         mov    cx,bx           ;LENFORM NUL bytes
         rep    stosb
         mov    di,offset init
         mov    byte ptr [di],MEDIA2
         mov    byte ptr [di+3],FATFOUR
         dec    cx
         mov    [di+1],cx       ;??FFFFh (FAT12 required up to 32 MB)
         mov    ax,LENSEC       ;DX:AX 0000:0200 (first FAT)
         neg    cx              ;CX 1 copy of LENFORM bytes at ES:DI
         call   near formit     ;F0FFFFh (medium F0h) start of FAT12
         mov    ax,LENSEC*(1+FATSEC)
         inc    cx              ;DX:AX 0000:1400 (1 + first 9 FATSEC)
         call   near formit     ;ditto for the following second FAT

        ;for LENFORM 64, 128, or 256 (pointless with LENFORM 512):
         mov    word ptr [di+LENFORM-2],MAGIC
         mov    [di],cx         ;MAGIC expected at end of MBR sector
         mov    [di+2],cx       ;clear FAT12 gibberish in the buffer
         mov    ax,LENSEC-LENFORM
         inc    cx              ;CX 1 copy of LENFORM bytes at ES:DI
         call   near formit     ;DX:AX 0000:01C0 (maybe, end of MBR)

        ;for LENFORM 64 (pointless with 128 or 256, fails if 512):
         mov    word ptr [di+48],13CDh  ;INT 13h for NDD 8.0 "check"
         mov    ax,bx           ;DX:AX @ LENFORM (2nd part of MBR)
         inc    cx              ;CX 1 copy of LENFORM bytes at ES:DI
         call   near formit     ;code permits LENFORM 64, 128, or 256

         mov    di,offset mbr2  ;the real thing, first LENFORM bytes
         mov    ax,cx           ;DX:AX 0000:0000 (1st part of MBR)
         inc    cx              ;CX 1 copy of LENFORM bytes at ES:DI
         call   near formit     ;copy begin of MBR sector
         call   newlabel
         ret                    ;CX = 0 (okay), return to init

formit:  push   cx              ;store number of copies
         mov    cx,bx           ;from ES:DI to XMS DX:AX
         call   xmsio
         jcxz   formerr         ;Z: program or XMS error
         pop    cx              ;reset number of copies
         add    ax,bx           ;BX = LENFORM
         adc    dx,0
         loop   formit
         retn                   ;near inline proc

formerr: pop    cx              ;trash pushed CX
         pop    cx              ;trash near return (CX <> 0)
         ret                    ;CX <> 0 error, return to init
format   endp

;-----------------------------------------------------------------------
; the remaining OVERLAY goes into the MBR (sector0) as indicated above

         org    OVERLAY
         assume ds:DGROUP, ss:DGROUP

; device functions 04h/08h/09h (input/output/output with verify)

input    proc   near            ;AX 0, DX subunit, ES:DI request
         inc    bp              ;BP 1 (read) disk data to DOS
write    label  near            ;BP 0 (write) set by intentry
         mov    cl,FIRST+1      ;expect word at ES:[DI+FIRST]
         call   chkbuf
         mov    bx,es:[di+FIRST]
         mov    si,es:[di+COUNT]
         lea    cx,[bx+si]
         cmp    cx,bx
         jb     secterr
         cmp    cx,nomemsec     ;subunit 0 sectors
         ja     secterr
         or     dx,dx           ;subunit ?
         jz     move64K
         cmp    cx,XMSSEC       ;subunit 1 sectors
         ja     secterr
         call   chkout          ;write access check (DX = 1, BP r/w)

move64K: cmp    si,65536/LENSEC
         ja     secterr         ;maximal 64 KB
         or     dx,dx           ;subunit ?
         mov    dx,si           ;DX count
         xchg   ax,bx           ;AX start
         les    di,es:[di+BUFFER]
         jz     tracker         ;subunit 0 i/o (.386 jump)
         jmp    tracxms         ;subunit 1 i/o

secterr: mov    es:[di+COUNT],ax
         jmp    badsect         ;sector not found
input    endp

;-----------------------------------------------------------------------
; device functions 0Dh/0Eh open/close device (OCR attribute bit)

close    proc   near            ;AX 0, DX subunit, ES:DI request
         neg    dx              ;DX 1 to -1, subunit DX 0 ignored
open     label  near            ;DX 1 as is, subunit DX 0 ignored
         add    opened2,dx      ;+/-1, overflow/underflow ignored
         ret                    ;CAVEAT: opened2 visible in devbpb2
close    endp

;-----------------------------------------------------------------------
; R/W access on boot sector, FAT, backup FAT, or root directory:
; ES:DI buffer, AX start sector below SYSSEC, DX count, BP r/w.
; ES:DI, AX, DX adjusted after copying system sectors, BP kept.

sysio    proc   near
         xor    bx,bx
         xchg   ax,bx           ;BX start sector, AX is 0
         or     dx,dx           ;DX remaining i/o sectors
         jz     sysokay         ;Z: remaining sectors DX 0
         or     bx,bx
         jnz    sys1            ;skip sector 0
         mov    si,offset sector0
         mov    cx,LENSEC/2
         call   sysmove
         inc    bx              ;adjust BX = 1
         dec    dx
         jz     sysokay         ;Z: remaining sectors DX 0
sys1:    cmp    bx,1
         jnz    sys2            ;skip sector 1
         call   sysfat          ;yields BX = 2
         jz     sysokay         ;Z: remaining sectors DX 0
sys2:    cmp    bx,2
         jnz    sys3            ;skip sector 2
         call   sysfat          ;yields BX = 3
         jz     sysokay         ;Z: remaining sectors DX 0
sys3:    mov    si,offset sector3
         mov    cx,USEDIR/2
         call   sysmove         ;copy used part of directory
         inc    bx              ;adjust BX = 4
         dec    dx
         or     bp,bp           ;0 write, 1 read
         jz     sysokay         ;skip write for virtual rest
         mov    cx,(LENDIR-USEDIR)/2
         call   sysmove         ;copy read only dir. entries
         mov    cx,(LENSEC-LENDIR)/2
         rep    stosw           ;fill buffer with AX 0
sysokay: xchg   ax,bx           ;AX = 4 (SYSSEC) if DX not 0
         ret                    ;DX remaining i/o sectors
sysio    endp

sysfat   proc   near
         mov    si,offset sector1
         mov    cx,USEFAT/2     ;USEFAT must be even (6,12,...)
         or     bp,bp
         jz     sysfat1         ;skip write for system entries
         call   sysmove         ;allow read for system entries
         jmp    short sysfat2
sysfat1: shl    cx,1            ;CX words to bytes
         add    si,cx           ;skip USEFAT bytes in DS:SI
         add    di,cx           ;skip USEFAT bytes in ES:DI

sysfat2: mov    cx,(LENFAT-USEFAT)/2
         call   sysmove
         or     bp,bp           ;0 write, 1 read
         jz     sysfat3         ;skip write for virtual rest
         mov    cx,(LENSEC-LENFAT)/2
         rep    stosw           ;fill buffer with AX 0
sysfat3: inc    bx              ;adjust BX
         dec    dx              ;adjust DX
         ret                    ;Z: DX is 0 (ready)
sysfat   endp

;-----------------------------------------------------------------------
genlook  proc   near            ;generic IOCTL function list search
         mov    bx,es:[di+GENF] ;BL major, BH minor function number
         cmp    bl,8            ;AX = 0 and BP = 0 set by intentry
         jnz    genzero         ;BL = 48h for FAT32 not supported
         mov    bl,00100000b    ;bit 5 (query) saved as BP 0 or 1
         test   bh,bl           ;bit 5 set ?
         jz     genout          ;BP 0 for 4x,5x, 0x,1x, Cx,Dx, 8x,9x
         inc    bp              ;BP 1 for 6x,7x, 2x,3x, Ex,Fx, Ax,Bx
         xor    bh,bl           ;bit 5 clear

genout:  mov    cx,MAXIOCTL     ;subunit 0 has no format functions
         mov    si,offset genlist-3
         or     dx,dx           ;subunit DX
         jz     genloop
         mov    cl,MAXIOCT2     ;subunit 1 supports generic format
         mov    si,offset genlis2-3

genloop: add    si,3            ;search minor function BH in list:
         cmp    bh,[si]         ;code byte + function offset word,
         loopne genloop         ;function offset follows code byte

genzero: ret                    ;return (N)Z function (not) found
genlook  endp

        rept    (offset message - $)
         int    3               ;fill rest of OVERLAY with CCh
        endm
.errnz   (offset message - $)   ;OVERLAY too big, move some code

;-----------------------------------------------------------------------
_TEXT    ends

         public request         ;public symbols for WDISasm listing
         public running         ;intentry semaphore
         public newstack
         public oldstack

         public strategy        ;device request (stored in request)
         public intentry        ;devive interrupt (execute request)
         public xmsdummy        ;far nop used if HMA+XMS unavailable

         public commands        ;device commands table:
         public badcmd          ;fail
         public getset          ;get/set logical device
         public fixed           ;media removable query
         public check           ;removable media change
         public build           ;build BPB
         public input
         public write
         public open
         public close
         public ioctlrd         ;IOCTL read
         public ioctlwr         ;IOCTL write
         public generic         ;IOCTL command
         public gentest         ;IOCTL availability
         public genlook         ;used by generic + gentest
         public tracker         ;used by input + write + trackio
         public tracxms         ;used by input + write + tracker
         public newlabel        ;used by volser + format
         public chkbuf          ;used to check request buffer size
         public chkout          ;used to check subunit 1 write access
         public chktrack        ;used to check track and head number
         public chs2abs         ;convert CHS to abs. sector number
         public lesdibuf        ;get and check i/o buffer address
         public xmsio           ;used to read/write XMS RAM disk
         public memio           ;used to read/write 1st MB + HMA
         public sysio           ;used to read/write MBR+FATs+DIR
         public sysfat          ;used by sysio for subunit 0 FATs
         public sysmove         ;used by sysio + sysfat to read/write
         public swapbuf         ;used to swap i/o source/target

         public params          ;IOCTL 840/860 subunit 1
         public trackio         ;IOCTL 841/861 subunit 1+0
         public forver          ;IOCTL 842/862 subunit 1
         public volser          ;IOCTL 846/866 subunit 1+0
         public drvflag         ;IOCTL 847/867 subunit 1
         public senslock        ;IOCTL 848/868 subunit 1+0
         public eject           ;IOCTL 849/869 subunit 1
         public fdlock          ;IOCTL 84A/86A subunit 1

         public sector0         ;subunit 0 MBR
         public bpb             ;subunit 0 BPB
         public nomemsec        ;subunit 0 number of sectors
         public nomulsec        ;subunit 0 sectors per track
         public volabel         ;subunit 0 volume label struct
         public message         ;subunit 0 installation
         public xmsdiag         ;last XMS error code
         public label2          ;subunit 1 label copy+flag for check
         public params2         ;subunit 1 params
         public devbpb2         ;subunit 1 default device BPB
         public opened2         ;subunit 1 open/close counter
         public genlis2         ;subunit 1 IOCTL list
         public genlist         ;subunit 0 subset of genlis2
         public sublocks        ;subunit 0+1 lock counters
         public xmspara         ;XMS read/write parameter structure
         public xmsdata         ;XMS handle
         public xms             ;XMS API or xmsdummy pointer
         public bpbptr          ;subunit 0+1 BPB pointers

         public sector1         ;subunit 0 1st + 2nd FAT sector
         public sector3         ;subunit 0 root directory
         public volume1         ;subunit 0 1st dir. entry label
         public dskentry        ;subunit 0 2nd dir. entry image file
         public hmaentry        ;subunit 0 4th dir. entry image HMA
         public time1           ;subunit 0 2nd dir. entry time: DSK
         public time3           ;subunit 0 4th dir. entry time: HMA
         public date1           ;subunit 0 2nd dir. entry date: DSK
         public date2           ;subunit 0 3rd dir. entry date: RAM
         public date3           ;subunit 0 4th dir. entry date: HMA

         public paranoia        ;last HMA paragraph for disk editors
         public mbr2            ;subunit 1 MBR \  / resident master
         public bpb2            ;subunit 1 BPB  ><  copy, also used
         public volabel2        ;subunit 1 vol /  \ for init. format
         public bottom          ;stack bottom
         public stacktop        ;stack (end of resident code)

         public init            ;used once (not resident)
         public msgerr          ;format failure message (XMS error)
         public datetime        ;used by init: BCD RTC date + time
         public filetime        ;used by init: BCD to DOS timestamp
         public filedate        ;used by init + filetime
         public bcd2int         ;used by filetime + filedate (AAM 16)
         public biosdate        ;used by init: BIOS date to BCD date
         public format          ;used by init: format subunit 1

         end    device
