;-----------------------------------------------------------------------
; Installation: DEVICE[high]=[<drive>:][<path>\]WINDUMB.SYS
;-----------------------------------------------------------------------
; Creation: WASM -wx -we -q -3 -bt=DOS -mt WINDUMB
;           WDIS -l -fi WINDUMB
;           WLINK op q,map format dos com file WINDUMB name WINDUMB.SYS
; Untested: MASM WINDUMB && LINK WINDUMB && EXE2BIN WINDUMB WINDUMB.SYS
;-----------------------------------------------------------------------
; WINDUMB is a dummy DOS character device installed after HIMEM.SYS.
; WINDUMB uses the XMS API in HIMEM.SYS to create a single UMB (upper
; memory block) at A200 up to B7FF.  This is similar to EMM386 option
; i=A200-B7FF.  The RAM from A000 to BFFF is used as video memory, but
; video modes 0..6 need only B800 up to BFFF.  This covers text modes
; 0..3 and VESA text modes, but not mode 7 (monochrome text).
;
; Clearly WINDUMB cannot coexist with applications for other modes or
; task switchers modifying video memory below B800, e.g., the PC DOS 7
; DOSSHELL is incompatible with WINDUMB.  The PC DOS 7 command MODE
; apparently uses memory below A200 as buffer if available; therefore
; WINDUMB starts at A200 (MCB A1FF).  Modify A200 in UMBSEG and UMBTXT
; for experiments with other UMB start segments.
;
; WINDUMB is interesting where EMM386 is unavailable *AND* the video
; memory A000..B7FF exists, but is not used, e.g., in a normal NT VDM
; (virtual DOS machine).  If you need EMM386 to enable A000..B7FF you
; can also include i=A200-B7FF directly without WINDUMB.
;
; WINDUMB installs its resident code at the end of the UMB.  The code
; consists of three or five parts:
; - an INT 10h interrupt handler checks function 0 (set video mode),
;   VGA and VESA text modes (excl. mono mode 7) are okay, otherwise
;   the VDM is unconditionally terminated with a jump to FFFF:0000.
; - an INT 2Dh interrupt handler supports AMIS (Alternate Multiplex
;   Interrupt Specification).
; - an XMS extension implements or augments XMS functions 10h..12h to
;   manage the (single) UMB at A200h.
; - if WINDUMB finds no XMS API, e.g., if HIMEM.SYS is not installed,
;   it installs an interrupt 2Fh handler with functions 4300h (query
;   XMS) and 4310h (get XMS API entry point).
; - if WINDUMB finds no XMS API it installs its own (fake) XMS API.
;   Maybe define NO11 to replace XMS function 11 "move block" by a
;   dummy BADFUNC, e.g., use option -dNO11 for WASM.
;
; WINDUMB moves the XBDA (eXtended BIOS Data Area) into its UMB if the
; XBDA segment noted in 0040:000Eh (= 0000:040Eh) consists of at most
; 1 KB below segment A000h or starts at 9FC0h.  This is similar to the
; EMM386 option MOVEXBDAHI.
;                                               (2010, Frank Ellermann)
;-----------------------------------------------------------------------

.errndef __TINY__               ;DOS device drivers are tiny (DS == CS)
         .386                   ;allow .386 instructions, e.g., "setne"
DGROUP   group  _TEXT
_TEXT    segment use16 para public 'CODE'
         assume nothing
         assume cs:DGROUP

UMBSEG   equ    0A200h
UMBTXT   equ    "A200"          ;i=A200-B7FF

        ifndef  TEST
DEVNAME  equ    "UMB@",UMBTXT
UMBINIT  equ    1               ;initial UMB state free
        else
DEVNAME  equ    "TST@",UMBTXT
UMBINIT  equ    0               ;initial UMB state allocated
        endif

device   dd     -1              ;device link
         dw     8000h           ;attribute 15 character device
         dw     offset strategy ;device strategy  entry offset
         dw     offset intentry ;device interrupt entry offset
         db     DEVNAME         ;character device name

;-----------------------------------------------------------------------
         org    $-18            ;overwrite unused device header
umbsize  dw     1 dup (?)       ;below DEVNAME (muxid. product)
muxident db     16 dup (?)      ;8 manufacturer plus 8 product
         db     0               ;followed by 0..64 ASCIIZ
LENIDENT equ    ($ - offset muxident)
MUXVERS  equ    0004h           ;version 0.4 (cf. LOMEM.SYS)
KBMAGIC  equ    "KB",0,0EBh,0,0CBh      ;see AMIS specification

umbfree  db     UMBINIT         ;1: free, 0: allocated
oldxms   dd     offset xmsfake  ;old XMS entry address

CMD      equ    02              ;offset command
STATUS   equ    03              ;offset status code
BUFFER   equ    14              ;offset I/O buffer / device end address
CONFIG   equ    18              ;offset addr. init. string [BPB pointer]
INIERR   equ    23              ;offset addr. init. error flag (DOS 5+)

;-----------------------------------------------------------------------
newvid   proc   far
         jmp    short chkvid    ;this is actually a "KB" structure...
oldvid   dd     0               ;far address old interrupt 00h
         db     KBMAGIC         ;"KB" AMIS specification 3.6 compatible

jmpvid:  jmp    cs:[oldvid]
chkvid:  cmp    ax,4F02h        ;check set VESA mode
         je     vbevid
         or     ah,ah           ;check set video mode
         jnz    jmpvid          ;anything else forwarded to old handler
         mov    ah,al
         and    al,7Fh          ;clear keep video RAM bit
         cmp    al,7
         jnb    badvid          ;mode 7 etc. bad: video RAM below B800h
         or     al,ah           ;reset keep video RAM bit
         mov    ah,0
         jmp    jmpvid          ;modes 00..06h okay: video RAM at B800h

vbevid:  mov    ax,bx           ;BX = wanted VESA mode
         and    ah,7Fh          ;clear keep video RAM bit
         cmp    ax,0007h        ;text 0000..0006h okay
         jb     oemvid
         cmp    ax,0108h        ;mode 0007..0107h bad
         jb     badvid
         cmp    ax,010Ch        ;text 0108..010Ch okay
         jnbe   badvid          ;else bad
oemvid:  mov    ax,4F02h
         jmp    jmpvid

badvid:  int    3               ;break point (OP: CCh)
         jmp    far reboot      ;BIOS boot (FFFF:0000)
newvid   endp
         align  2

;-----------------------------------------------------------------------
newmux   proc   far
         jmp    short chkmux    ;this is actually a "KB" structure...
oldmux   dd     0               ;far address old interrupt 2Dh
         db     KBMAGIC         ;"KB" AMIS specification 3.6 compatible

jmpmux:  jmp    cs:[oldmux]
muxlist  db     10h
         dw     offset newvid
mux2Fh:  db     2Dh             ;2Fh (patched if installed)
         dw     offset newmux   ;offset new2Fh
         db     2Dh
         dw     offset newmux

MUXHAND  equ    ($+2)           ;AMIS MUX id. determined by init
chkmux:  cmp    ah,-1
         jne    jmpmux
         or     al,al           ;AMIS 0: installation test
         jnz    uninst
         not    al              ;AL 255 installed
         mov    cx,MUXVERS      ;version (CH major)
         mov    dx,cs           ;DX:DI identification
         mov    di,offset muxident
         iret                   ;-----------------------
uninst:  cmp    al,2            ;AMIS 2: unistall
         jnz    getlist
         mov    al,5            ;uninstall cannot work
         mov    bx,cs           ;(XMS hook not visible)
         iret                   ;-----------------------
getlist: cmp    al,4            ;AMIS 4: get hook list
         jnz    devinfo
         mov    dx,cs           ;DX:BX = MUX hook list
         mov    bx,offset muxlist
         iret                   ;-----------------------
devinfo: cmp    al,6            ;AMIS 6: device info
         jnz    retzero
         mov    ax,1
         mov    dx,cs           ;DX:BX = device header
         xor    bx,bx           ;device offset 0
         iret                   ;-----------------------
retzero: mov    al,0            ;otherwise set AL 0
         iret
newmux   endp

;-----------------------------------------------------------------------
; used XMS 2.0 error codes (BL) and function numbers (AH)

A20SET   equ    03h             ;fake XMS: A20  enable (owner)
A20CLR   equ    04h             ;fake XMS: A20 disable (owner)
A20STAT  equ    07h             ;fake XMS: A20 status
XMALLOC  equ    09h             ;fake XMS: allocate XMS memory
MEMMOVE  equ    0Bh             ;fake XMS: move DOS/XMS memory

BADFUNC  equ    080h            ;XMS code: unknown or not implemented
A20FAIL  equ    082h            ;XMS code: cannot modify A20 line
GENFAIL  equ    08Eh            ;XMS code: general failure
HMAFAIL  equ    090h            ;XMS code: HMA unavailable
HMAUSED  equ    091h            ;XMS code: no free HMA memory
HMAMINE  equ    092h            ;XMS code: HMAMIN prohibition
HMAFREE  equ    093h            ;XMS code: release HMA error
XMSUSED  equ    0A0h            ;XMS code: no free XMS memory
BADHAND  equ    0A2h            ;XMS code: invalid handle (A2..A4)
BADSIZE  equ    0A7h            ;XMS code: invalid length
OVERLAP  equ    0A8h            ;XMS code: source overlaps target
RAMFAIL  equ    0A9h            ;XMS code: RAM parity error

TOOMUCH  equ    0B0h            ;XMS code: smaller UMBs available
ALLUSED  equ    0B1h            ;XMS code: no more UMBs available
UMBSEGV  equ    0B2h            ;XMS code: invalid UMB segment
UMALLOC  equ    10h             ;XMS function: allocate UMB
RELEASE  equ    11h             ;XMS function: release UMB
REALLOC  equ    12h             ;XMS function: realloc UMB

;-----------------------------------------------------------------------
newxms   proc   far
         jmp    short chkxms    ;\  / start with EB 03 90 90 90
         nop                    ; \/  as per XMS specification:
         nop                    ; /\  extensions can then patch
         nop                    ;/  \ a far jump EA ?? ?? ?? ??
chkxms:  pushf
         cmp    ah,UMALLOC      ;AH 10: allocate UMB
         jb     jmpxms          ;other functions handled by XMS
         je     getumb
         cmp    ah,REALLOC      ;AH 12: realloc. UMB
         jnbe   jmpxms          ;other functions handled by XMS
         cmp    dx,UMBSEG
         jne    jmpxms          ;other UMBs handled by XMS
         cmp    cs:umbfree,0
         jne    umbmine         ;UMB not allocated
         cmp    ah,RELEASE      ;AH 11: release UMB
         je     relumb
         mov    ax,cs:umbsize
         cmp    bx,ax
         jbe    umbokay         ;allow any smaller size
         xchg   dx,ax
umbhuge: mov    bl,TOOMUCH      ;BL B0: smaller UMBs available
         jmp    short umbfail

umbnone: mov    bl,ALLUSED      ;BL B1: no more UMBs available
         jmp    short umbfail

umbmine: mov    bl,UMBSEGV      ;BL B2: invalid UMB segment
umbfail: xor    ax,ax           ;0: fail
         popf
         retf                   ;DOS expects unmodified flags

relumb:  inc    cs:umbfree      ;---- release UMB --------------
umbokay: mov    ax,1            ;1: okay
         popf
         retf                   ;DOS expects unmodified flags

jmpxms:  popf                   ;--- other XMS functions -------
         jmp    cs:[oldxms]     ;functions handled by XMS

getumb:  push   dx              ;DX = wanted size
         call   cs:[oldxms]
         or     ax,ax           ;NZ okay, Z fail
         pop    ax              ;AX = wanted size
         jnz    umbokay         ;old handler success
         cmp    cs:umbfree,0
         je     umbnone         ;old handler failure
         cmp    bl,TOOMUCH
         jne    fixumb          ;our UMB is available
         cmp    dx,cs:umbsize
         jnbe   umbhuge         ;bigger than our UMB
fixumb:  mov    dx,cs:umbsize
         cmp    dx,ax           ;DX available, AX wanted
         jb     umbhuge
         dec    cs:umbfree
         mov    bx,UMBSEG       ;allocated UMB segment
         jmp    umbokay         ;spec. allows to return more than wanted
newxms   endp

;-----------------------------------------------------------------------
ADJUST   =      $               ;fill paragraph with CCh (INT 3)
         align  10h             ;resident code end at paragraph
ADJUST   =      $ - ADJUST      ;new offset, align gap 00..0Fh
         org    $ - ADJUST      ;old offset, gap = gap mod 10h
        if      ADJUST mod 08h  ;fill gap to xxx8h or to xxx0h
         db     ADJUST mod 08h dup (0CCh)
        endif
        if      ADJUST mod 10h ge 8
         db     DEVNAME         ;upper gap from xxx8h to xxx0h
        endif
endcode  label  byte            ;resident end (if HIMEM.SYS found)

;-----------------------------------------------------------------------
new2Fh   proc   far             ;only installed if HIMEM.SYS not found
         jmp    short chk2Fh    ;this is actually a "KB" structure...
old2Fh   dd     0               ;far address old interrupt 2Fh
         db     KBMAGIC         ;"KB" AMIS specification 3.6 compatible

jmp2Fh:  jmp    cs:[old2Fh]
chk2Fh:  cmp    ax,4300h        ;XMS presence check
         jne    api2Fh
         or     al,80h
         iret

api2Fh:  cmp    ax,4310h        ;XMS API entry point
         jne    jmp2Fh
         push   cs
         pop    es
         mov    bx,offset newxms
         iret
new2Fh   endp

;-----------------------------------------------------------------------
xmsfake  proc   far             ;used by newxms if HIMEM.SYS not found
         pushf
         cld
         cmp    ah,A20STAT
         jbe    a20func         ;functions 1..7

         mov    al,BADFUNC      ;unknown subfunction
         cmp    ah,UMALLOC      ; or not implemented
         jnb    xmsblot         ;functions 16..255

         mov    al,XMSUSED      ;no XMS memory available
         cmp    ah,XMALLOC
         jnb    xmsfunc         ;functions 9..15
         cwd                    ;size ext. mem. (XMS function 08)
xmsfunc: jbe    xmsblot         ;functions 8..9

         mov    al,BADHAND      ;invalid handle (there are no handles)
         cmp    ah,MEMMOVE      ;function 11
         jne    xmsblot         ;function 10 or 12..15 (BL = A2h)
         inc    ax              ;invalid source handle (BL = A3h)
         cmp    word ptr [si+04],0
         jne    xmsblot
         inc    ax              ;invalid target handle (BL = A4h)
         cmp    word ptr [si+10],0
         jne    xmsblot

         call   xmsmove         ;DS:SI = XMS move structure
         or     al,al
         jz     xmsokay
xmsblot: mov    bl,al           ;BL = error code
         jmp    short xmsfail

a20func: cli                    ;no interrupt until popf/retf
         push   ds
         push   bx
         push   cs
         pop    ds
         assume ds:DGROUP
         mov    bx,offset a20cmds
         mov    al,ah           ;AL = function 0..7
         xlat                   ;AL = offset -128..+127 \/ please check
         cbw                    ;AX = offset -128..+127 /\ XLAT a20cmds
         add    bx,ax           ;BX = offset of function 0..7
         mov    ax,word ptr hmabyte
         jmp    bx              ;AL = hmabyte, AH = a20byte

a20f00:  pop    bx              ;function 0: XMS version
         pop    ds              ;reset DS for function 0
         xor    dx,dx           ;0: HMA unavailable
         cmp    al,-1
         setne  dl              ;1: HMA exists
         mov    ax,0200h        ;XMS version 2.0
         mov    bx,MUXVERS      ;LOMEM.SYS 0.4 (same as WINDUMB.SYS)
         jmp    short xmsexit

a20f01:  pop    bx              ;function 1: acquire HMA
         mov    ah,HMAUSED
         xor    al,1            ;AL 1 (free) to 0 (used)
         jnz    hmabad1         ;HMA used or unavailable
        ifdef   HMAMIN_SET      ;-----------------------------------
         mov    ah,HMAMINE
         cmp    dx,hmamin
         jb     hmabad2         ;HMAMIN prohibition
        endif                   ;-----------------------------------
hmaokay: mov    hmabyte,al
popokay: pop    ds              ;reset DS for functions 1..7 (okay)
xmsokay: mov    ax,1            ;AX 1: okay
xmsexit: popf
         retf

a20f02:  pop    bx              ;function 2: release HMA
         mov    ah,HMAFREE
         xor    al,1            ;AL 0 (used) to 1 (free)
         jz     hmabad2         ;HMA not allocated
         jns    hmaokay
hmabad1: jns    hmabad2
         mov    ah,HMAFAIL      ;HMA unavailable (-1)
hmabad2: mov    bl,ah
popfail: pop    ds              ;reset DS for functions 1..7 (fail)
xmsfail: xor    ax,ax
         popf
         retf

        ifdef   HMAMIN_SET      ;-----------------------------------
hmamin   dw     0               ;HMAMIN limit, set by init (not yet)
        endif                   ;-----------------------------------
hmabyte  db     1               ;1: free, 0: used, -1: bad \/ keep this
a20byte  db     0               ;A20 off, adjusted by init /\ together
a20cmds  db     offset a20f00 - offset a20cmds
         db     offset a20f01 - offset a20cmds
         db     offset a20f02 - offset a20cmds
         db     offset a20f03 - offset a20cmds
         db     offset a20f04 - offset a20cmds
         db     offset a20f05 - offset a20cmds
         db     offset a20f06 - offset a20cmds
         db     offset a20f07 - offset a20cmds

a20f03:  mov    ah,-1           ;function 3: A20 owner  enable
         jmp    short a20mod
a20f04:  mov    ah,0            ;function 4: A20 owner disable
         jmp    short a20mod
a20f05:  inc    ah              ;function 5: A20 local  enable
         jnz    a20mod
         jmp    short a20lim    ;undo  overflow
a20f06:  or     ah,ah           ;function 6: A20 local disable
         jz     a20mod          ;skip underflow
a20lim:  dec    ah

a20mod:  in     al,92h          ;port 92h A20 bit 1 in AL
         mov    bl,al           ;AH is intended A20 state
         or     al,2            ;set bit 1 (to enable A20)
         or     ah,ah           ;Z: disable A20 (wrap around)
         jnz    a20_on          ;NZ: enable A20 (use HMA)

         xor    al,2            ;clear bit 1 (disable A20)
         call   near a20chk
         jne    a20okay         ;wrap around is okay
a20bad:  pop    bx              ;BH of caller reset
         mov    bl,A20FAIL
         jmp    popfail

a20_on:  call   near a20chk
         jne    a20bad          ;wrap around is wrong
a20okay: mov    a20byte,ah
         pop    bx              ;BX of caller reset
         jmp    popokay

a20f07:  call   a20wrap         ;function 7: A20 status
         pop    bx              ;BH of caller reset
         mov    bl,0            ;BL 0: no error
         je     popokay         ;AX 1: A20 enabled
         jne    popfail         ;AX 0: wrap around

a20chk:  cmp    al,bl
         je     a20tst          ;skip if AL = BL
         out    92h,al
a20tst:  call   a20wrap
         retn                   ;near inline proc
xmsfake  endp

;-----------------------------------------------------------------------
         assume ds:nothing

a20wrap  proc   near            ;modifies BX, expects CLI
         push   es
         push   ds
         push   ax
         mov    bx,-1           ;ES = FFFFh
         mov    es,bx
         inc    bx              ;DS = 0000h
         mov    ds,bx
         mov    al,[bx]         ;get DS:[00h]
         mov    ah,es:[bx+10h]  ;get ES:[10h]
         not    byte ptr [bx]   ;new DS:[00h]
         cmp    ah,es:[bx+10h]
         mov    [bx],al         ;old DS:[00h]
         pop    ax
         pop    ds
         pop    es
         ret                    ;NE: A20 off, E: on
a20wrap  endp

;-----------------------------------------------------------------------
xmsmove  proc   near            ;DS:SI = XMS move structure
        ifndef  NO11            ;------ use int 15h to move block ------
         pusha
         push   es
         push   ds
         push   ss
         pop    es
         assume ds:nothing, es:nothing, ss:nothing

         mov    bp,sp           ;ES:BP = SS:SP
         mov    cx,48+8         ;48 for int 15h, +8 (2 dwords)
         sub    sp,cx           ;SS:SP alloca CX
         xor    ax,ax
         mov    di,sp           ;ES:DI alloca
         rep    stosb           ;clean alloca CX
         mov    di,sp           ;ES:DI alloca
         push   bp              ;store old SP

         mov    bp,48           ;DS:[SI+04] source handle 0
         mov    bx,6            ;DS:[SI+06] source seg:ofs
         call   near linear     ;ES:[DI+48] source linear
         mov    bp,52           ;DS:[SI+10] target handle 0
         mov    bx,12           ;DS:[SI+12] target seg:ofs
         call   near linear     ;ES:[DI+52] target linear

         mov    cx,[si+0]       ;= linear length low
         mov    bp,[si+2]       ;= linear length high
         push   ss
         pop    ds              ;DS = ES = SS (stack of caller)

         mov    ax,[di+48]      ;= linear source low
         mov    bx,[di+50]      ;= linear source high
         add    ax,cx           ;+ linear length low
         adc    bx,bp           ;+ linear length high
         jc     oddmove         ;lost bit, bad length
         shr    cx,1            ;length < 8000h words
         jc     oddmove         ;lost bit, bad length
         inc    bp              ;64 KB blocks + first
         jz     oddmove         ;lost bit, bad length

         mov    dx,[di+54]      ;= linear target high
         cmp    bx,dx           ;(src+len) ? dst high
         mov    bx,[di+52]      ;= linear target low
         jb     trymove         ;(src+len) < dst ok.
         ja     chkmove         ;(src+len) > dst unclear
         cmp    ax,bx           ;(src+len) ? dst low
         jna    trymove         ;(src+len) > dst unclear

chkmove: cmp    dx,[di+50]      ;target ? source high
         jb     trymove         ;target < source ok.
         ja     badmove         ;target > source bad
         cmp    bx,[di+48]      ;target ? source low
         ja     badmove         ;target > source bad

trymove: mov    si,di           ;ES:SI = int 15h table
         mov    di,93h          ;int 15h access rights
nxtmove: push   bp
         push   bx
         mov    bp,48           ;linear source
         mov    bx,16           ;int 15h table: source
         call   near setmove
         mov    bp,52           ;linear target
         mov    bx,24           ;int 15h table: target
         call   near setmove
         pop    bx
         pop    bp
         pushf                  ;-----------------------
         sti
         mov    ah,87h
         popf                   ;-----------------------
         int    15h             ;move block (AH 87h)
         cmp    ah,3
         je     a20move         ;A20 failure
         cmp    ah,1
         je     parmove         ;memory parity error
         ja     intmove         ;interrupted move (?)
         mov    cx,8000h        ;next 64 KB (words)
         dec    bp              ;rest 64 KB blocks
         jnz    nxtmove         ;next block

         pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,0            ;okay
         ret                    ;-----------------------
oddmove: pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,BADSIZE      ;odd or invalid length
         ret                    ;-----------------------
badmove: pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,OVERLAP      ;source overlaps target
         ret                    ;-----------------------
parmove: pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,RAMFAIL      ;move block parity error
         ret                    ;-----------------------
a20move: pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,A20FAIL      ;move block A20 failure
         ret                    ;-----------------------
intmove: pop    sp              ;undo alloca
         pop    ds
         pop    es
         popa
         mov    al,GENFAIL      ;move block interrupted
         ret                    ;-----------------------

setmove  label  near            ;SS = DS, src/dst [SI+BP]
         mov    ax,[si+bp+0]    ;= linear src/dst low
         mov    dx,[si+bp+2]    ;= linear src/dst high
         mov    word ptr [si+bx+0],-1   ;max. length 64 KB
         mov    [si+bx+2],ax    ;int 15h: src/dst low
         mov    [si+bx+4],dl    ;int 15h: src/dst high (DL)
         mov    [si+bx+5],di    ;int 15h: access rights
         mov    [si+bx+7],dh    ;int 15h: src/dst high (DH)
         add    ax,cx           ;+ linear length words
         adc    dx,0
         add    ax,cx           ;+ linear length bytes
         adc    dx,0
         mov    [si+bp+0],ax    ;set next src/dst low
         mov    [si+bp+2],dx    ;set next src/dst high
         retn                   ;near inline proc

linear   label  near            ;SS = ES, linear SS:[DI+BP]
         push   [si+bx+0]       ;src/dst offset
         pop    [di+bp+0]       ;src/dst address low start
         mov    ax,[si+bx+2]    ;src/dst segment
         rol    ax,4            ;4321 to 3214
         mov    bx,ax           ;3214 (example)
         and    bx,000Fh        ;0004 (segment bits 12..15)
         xor    ax,bx           ;3210 (16bit offset)
         add    [di+bp+0],ax    ;linear address low  final
         adc    [di+bp+2],bx    ;linear address high
         retn                   ;near inline proc
        else                    ;---------------------------------------
         mov    bl,BADFUNC      ;32bit real memory move not implemented
         ret                    ;(only) this violates XMS specification
        endif                   ;---------------------------------------
xmsmove  endp

;-----------------------------------------------------------------------
ADJUST   =      $               ;fill paragraph with CCh (INT 3)
         align  10h             ;resident code end at paragraph
ADJUST   =      $ - ADJUST      ;new offset, align gap 00..0Fh
         org    $ - ADJUST      ;old offset, gap = gap mod 10h
        if      ADJUST mod 08h  ;fill gap to xxx8h or to xxx0h
         db     ADJUST mod 08h dup (0CCh)
        endif
        if      ADJUST mod 10h ge 8
         db     DEVNAME         ;upper gap from xxx8h to xxx0h
        endif
endxms   label  byte            ;resident end (if HIMEM.SYS not found)

;------------------------------- non-resident section ------------------
; Serialization strategy:  Each request is initialized with status busy.
; If the old request offset is -1 (invalid) a new request can be noted.
; The request offset is invalidated (-1) at the begin of intentry.  If
; the request offset is invalid intentry exits.  A "running" semaphore
; is incremented at the start of intentry and decremented at exit.  The
; initial running value is -1 and permits intentry to process a request.

request  dd     -1              ;request block address pointer
running  db     -1

strategy proc   far
         pushf
         cli
         mov    word ptr es:[bx+STATUS],200h    ;status busy, not ready
         cmp    word ptr cs:request+0,-1        ;at most one request
         jne    intbusy
         mov    word ptr cs:request+0,bx        ;ES:BX is address...
         mov    word ptr cs:request+2,es        ;...of request block
intbusy: popf
         retf
strategy endp

;-----------------------------------------------------------------------
intentry proc   far
         pushf
         pusha                  ;save registers
         push   ds
         push   es
         cld
         push   cs
         pop    ds              ;data in code segment
         assume ds:DGROUP

         inc    byte ptr running
         jnz    intnop
         les    di,request      ;ES:DI request
         mov    si,-1           ;invalidate request
         cmp    si,di
         je     intnop          ;no pending request
         mov    word ptr request,si

         mov    ax,8003h        ;unknown request
         cmp    byte ptr es:[di+CMD],0
         jne    intbad          ;support only init (0)
         mov    ax,8005h        ;invalid request size
         cmp    byte ptr es:[di],INIERR
         jbe    intbad          ;INIERR asserts DOS 5+
         call   init

intbad:  or     ah,1            ;status ready bit
         mov    es:[di+STATUS],ax
intnop:  dec    byte ptr running
         pop    es              ;restore registers
         pop    ds
         popa
         popf
         retf
intentry endp

;-----------------------------------------------------------------------
init     proc   near            ;device initialization
         push   es              ;preserve ES:DI request
         push   di
         inc    si              ;cheap 0 (resident size)
         mov    es:[di+BUFFER+0],si
         mov    es:[di+BUFFER+2],cs

         push   ds
         pop    es              ;overwrite unused device header:
         xor    di,di           ;ES:DI = CS:0
         mov    si,offset umbbot+3
         mov    cx,1+4          ;1 word initial umbsize (device+0)
         rep    movsw           ;4 words begin muxident (device+2)

         call   muxtest
         mov    dx,offset msgmux
         jz     initerr         ;already installed

         mov    ah,0Fh          ;assert video mode 0..6
         int    10h
         cmp    al,7            ;exclude mode 7 (mono text)
         mov    dx,offset msgvid
         jnb    initerr

         mov    ax,UMBSEG-1
         mov    si,offset umbbot
         call   umbsign         ;MCB before UMB
         mov    dx,offset msgbot
         jne    initerr         ;cannot use Axxx video memory

         mov    ax,0B7FFh
         mov    si,offset umbend
         call   umbsign         ;MCB after UMB
         mov    dx,offset msgumb
         jne    initerr         ;cannot use mono video memory

         mov    ax,4300h        ;test XMS presence
         int    2Fh
         cmp    al,80h
         jne    initxms

         mov    dx,offset msgjmp
         mov    ax,4310h        ;get XMS entry point
         int    2Fh
findxms: mov    ax,es:[bx]
         cmp    ax,03EBh        ;short jump is first
         jz     hookxms
         cmp    al,0EAh         ;far jump is hooked
         jnz    initerr         ;bad code, give up
         les    bx,es:[bx+1]
         jmp    findxms

initerr: call   initmsg
         pop    di              ;restore ES:DI request
         pop    es              ;report CONFIG.SYS line (DOS 5+)
         inc    byte ptr es:[di+INIERR]
         mov    ax,800Ch        ;general failure
         ret

hookxms: lea    ax,[bx+5]       ;skip far jump (5 bytes)
         mov    word ptr oldxms+0,ax
         mov    word ptr oldxms+2,es
         mov    cx,offset endcode
         call   movcode         ;CX in: length, out: segment
         mov    dx,offset msgass
         jnz    initerr         ;movcode error (unexpected)
         mov    byte ptr es:[bx+0],0EAh
         mov    word ptr es:[bx+1],offset newxms
         mov    word ptr es:[bx+3],cx
         jmp    initmux         ;CX = copied code segment

initxms: mov    bx,offset mux2Fh
         mov    byte ptr [bx+0],2Fh
         mov    word ptr [bx+1],offset new2Fh
         mov    cx,offset endxms
         call   movcode         ;CX in: length, out: segment
         mov    dx,offset msgass
         jnz    initerr         ;movcode error (unexpected)

         push   ds              ;CX = copied code segment
         mov    ds,cx           ;segment fixup for xmsfake
         mov    word ptr oldxms+2,ds
         mov    ah,A20STAT      ;XMS function 7: A20 status
         call   [oldxms]
         mov    bx,A20SET + A20CLR * 256
         or     al,ah
         jz     initest         ;check enable + disable
         dec    a20byte         ;initialize enabled A20
         xchg   bh,bl           ;check disable + enable

initest: mov    ah,bl           ;XMS function 3: A20 enable
         call   [oldxms]        ;(or function 4 if swapped)
         or     ah,al
         jz     initnak
         mov    ah,bh           ;XMS function 4: A20 disable
         call   [oldxms]        ;(or function 3 if swapped)
         or     ah,al
         jz     initnak

         pop    ds              ;DS reset for DS:msga20
         cmp    bl,bh
         jb     initack         ;enable + disable (okay)
         mov    dx,offset msga20
         call   initmsg         ;disable + enable (unusual)
         jmp    initack         ;CX = copied code segment

initnak: neg    hmabyte         ;-1: HMA unavailable
         pop    ds              ;DS reset for DS:msghma
         mov    dx,offset msghma
         call   initmsg         ;CX = copied code segment

initack: mov    al,2Fh          ;----------------------------
         mov    si,offset new2Fh
         call   setint          ;set new mux. INT 2Fh
initmux: mov    al,10h          ;----------------------------
         mov    si,offset newvid
         call   setint          ;set new video interrupt
         mov    al,2Dh          ;----------------------------
         mov    si,offset newmux
         call   setint          ;set new AMIS INT 2Dh

         pop    di              ;restore ES:DI request
         pop    es
         xor    ax,ax
         ret                    ;AX = 0 (okay)
init     endp

;-----------------------------------------------------------------------
initmsg  proc   near
         mov    ah,9
         push   dx
         mov    dx,offset msgbeg
         int    21h
         pop    dx
         int    21h
         mov    dx,offset msgend
         int    21h
         ret
initmsg  endp

;-----------------------------------------------------------------------
setint   proc   near
         push   ds
         mov    ah,35h          ;fetch ES:BX = address old handler
         int    21h
         mov    ds,cx           ;DS     = CX = segment new handler
         mov    dx,si           ;DX     = SI =  offset new handler
         mov    [si+2],bx       ;[SI+2] = BX =  offset old handler
         mov    [si+4],es       ;[SI+4] = ES = segment old handler
         mov    ah,25h
         int    21h             ;install new int AL handler DS:DX
         pop    ds
         ret
setint   endp

;-----------------------------------------------------------------------
muxtest  proc   near            ;modifies ES:DI, SI, AX, BX, CX, DX
         mov    bx,0FF00h       ;MUX installation check
muxloop: mov    ah,bl
         mov    al,0
         int    2Dh
         or     al,al           ;free handle if AL 0
         jz     muxfree
         mov    si,offset muxident
         mov    es,dx           ;DX:DI identification
         mov    cx,LENIDENT
         repe   cmpsb
         jne    muxnext
         ret                    ;Z: already installed

muxfree: cmp    bl,bh
         jnb    muxnext         ;skip if not first free
         mov    bh,bl           ;note first free handle
muxnext: inc    bl              ;next handle BL (1..255)
         jnz    muxloop
         mov    MUXHAND,bh      ;use 1st MUX handle
         inc    bh              ;Z: no free MUX handle
         ret                    ;NZ: okay (handle set)
muxtest  endp

;-----------------------------------------------------------------------
movhigh  proc   near
         assume ds:nothing, es:nothing
         mov    ax,cx           ;returns UMB DX, AX = 0, DS = CS
         dec    ax              ;modifies ES:DI, SI, BX, CX
         shr    ax,4
         inc    ax              ;CX = length, AX = paragraphs

         mov    bx,cs:umbsize
         sub    bx,ax
         mov    dx,UMBSEG-1
         mov    es,dx           ;ES = 1st MCB before UMBSEG
         add    dx,bx
         push   dx              ;DX = new MCB
         inc    dx              ;DX = new UMB
         dec    bx              ;BX = new size (excl. MCB)
         mov    cs:umbsize,bx
         mov    es:[03],bx
         mov    es,dx           ;ES = new UMB
         xor    di,di           ;ES:DI = target
         rep    movsb           ;DS:SI = source (input)

         push   cs
         pop    ds
         assume ds:DGROUP
         mov    si,offset umbbot
         mov    [si+3],ax
         pop    ax              ;AX = new MCB

umbsign  label  near            ;modifies ES:DI, SI, AX, CX
         mov    cx,5
         mov    es,ax           ;ES = MCB segment (input AX)
         xor    di,di           ;SI = signature offset
         rep    movsb           ;copy MCB struct for MCB walkers
         xor    ax,ax           ;AX = 0
         stosw
         stosb
         mov    cl,4
         mov    si,offset muxident + 8
         push   si
         push   di
         rep    movsw           ;fill MCB paragraph with DEVNAME
         mov    cl,4
         pop    di
         pop    si
         repe   cmpsw
         ret                    ;Z: okay, NZ: read only memory
movhigh  endp

;-----------------------------------------------------------------------
movcode  proc   near            ;modifies DI, SI, AX, CX, DX
         push   es              ;keeps ES:BX
         push   bx
         push   ds
         push   cx              ;CX = length of code
         xor    si,si           ;----- XBDA move -----
         mov    ds,si
         assume ds:SEGNULL
         mov    ax,[xbdaseg]
         cmp    ah,0A0h         ;check XBDA AX < 0A000h
         jnb    oddxbda         ;already moved
         mov    ds,ax           ;DS:SI XBDA source
         assume ds:nothing
         cmp    ax,9FC0h        ;1 KB: XBDA AX = 09FC0h
         je     stdxbda
         cmp    byte ptr [si],2 ;check XBDA size [DS:0]
         jnb    oddxbda         ;ignore more than 1 KB

stdxbda: mov    cx,1024
         call   movhigh         ;DX = UMB of copied XBDA
         stc                    ;AX = 0, Z okay, NZ fail
         jnz    oddxbda
         mov    ds,ax
         assume ds:SEGNULL
         mov    [xbdaseg],dx    ;moved XBDA segment
         clc                    ;NC okay or no copy (jnb)

oddxbda: pop    cx              ;----- code move -----
         pop    ds              ;reset DS:CX
         assume ds:DGROUP
         jc     movfail
         xor    si,si           ;source DS:SI, length CX
         call   movhigh         ;DX = UMB of copied code
         xchg   cx,dx           ;CX = UMB of copied code
movfail: pop    bx
         pop    es              ;reset ES:BX
         ret                    ;AX = 0, Z okay, NZ fail
movcode  endp

;-----------------------------------------------------------------------
umbbot   db     "M"                     ;\/ M: MCB copied to UMBSEG-1 --
         dw     UMBSEG, 0B7FFh-UMBSEG   ;/\ copied to device+0 header \/
         db     "Frank E "              ;-- copied to device+2 header /\
umbend   db     "Z"                     ;\/ Z: MCB copied to 0B7FFh
         dw     0B800h, 0FFFFh-0B800h   ;/\ (for debug + MCB walkers)

msgvid   db     "unsupported video mode$"
msgjmp   db     "cannot hook HIMEM.SYS$"
msgbot   db     "cannot access segment ",UMBTXT,"-1$"
msgumb   db     "cannot access segment B7FF$"   ;\/ msgbot should cover
msgass   db     "assertion failed$"             ;/\ msgass access error
msgmux   db     "already installed$"
msga20   db     "A20 line not disabled$"
msghma   db     "HMA unavailable$"

msgbeg   db     DEVNAME,": $"
msgend   db     13,10,"$"

;------------------------------- system data ---------------------------
_TEXT    ends
SEGNULL  segment use16 at 0 private 'FAR_BSS'   ;improve WDISasm listing
         org    40Eh
xbdaseg  dw     1 dup (?)       ;XBDA (eXtended BIOS Data Area)
SEGNULL  ends
SEGBOOT  segment use16 at -1
reboot   label  byte
SEGBOOT  ends

         public strategy        ;device request (stored in request)
         public intentry        ;devive interrupt (execute request)
         public init
         public request         ;public symbols for WDISasm listing
         public running         ;-1: inactive
         public xbdaseg
         public umbfree         ;1: free, 0: allocated (single UMB)
         public umbsize         ;0B800h -UMBSEG-1 -code-1 (-XBDA-1)
         public movcode         ;move XBDA + code to B7xx UMBs
         public movhigh         ;copy data to UMB, create MCB
         public umbsign         ;MCBs free UMB, code, XBDA, B800
         public muxident        ;AMIS ident. (installation check)
         public muxtest         ;AMIS installation

         public newmux          ;int 2Dh (Alt. Multiplex Int. Spec.)
         public oldmux
         public newvid          ;int 10h (video)
         public oldvid
         public newxms          ;XMS API entry point (HIMEM.SYS)
         public oldxms

         end    device
